<?php

namespace App\Providers;

use App\Entities\Master;
use App\Entities\Nati\Aliquota;
use App\Entities\Nati\Categoria;
use App\Entities\Nati\FormaPagamento;
use App\Entities\Nati\Measure;
use App\Entities\Nati\Customer;
use App\Entities\Nati\Product;
use App\Entities\Nati\ProdutoConfig;
use App\Entities\Nati\Sector;
use App\Entities\Permission;
use App\Entities\PontoDelivery;
use App\Entities\Profile;
use App\Entities\Supplier;
use App\Entities\User;
use App\Observers\AliquotaObserver;
use App\Observers\CategoriaObserver;
use App\Observers\CustomerObserver;
use App\Observers\FormaPagamentoObserver;
use App\Observers\MasterObserver;
use App\Observers\MedidaObserver;
use App\Observers\PermissionObserver;
use App\Observers\PontoDeliveryObserver;
use App\Observers\ProductObserver;
use App\Observers\ProdutoConfigObserver;
use App\Observers\ProfileObserver;
use App\Observers\SetorObserver;
use App\Observers\SupplierObserver;
use App\Observers\UserObserver;
use Carbon\Carbon;
use Illuminate\Http\Resources\Json\JsonResource;
use Illuminate\Support\ServiceProvider;
use Laravel\Passport\Passport;
use App\Entities\GestorConfig;
use App\Helpers\CryptoHelper;
use App\Services\Gestor\GestorAuthService;
use App\Services\Gestor\GestorIntegrationService;
use Illuminate\Support\Facades\Log;


class AppServiceProvider extends ServiceProvider
{
    /**
     * Register any application services.
     *
     * @return void
     */
    public function register()
    {
        // Registrar outros serviços antes
        $this->app->register(RepositoryServiceProvider::class);

        // Registrar os singletons SEM acessar o banco ainda
        $this->app->singleton(GestorAuthService::class, function ($app) {
            return new GestorAuthService(null, null, null); // Inicialmente vazio
        });

        $this->app->singleton(GestorIntegrationService::class, function ($app) {
            return new GestorIntegrationService($app->make(GestorAuthService::class));
        });
    }

    /**
     * Bootstrap any application services.
     *
     * @return void
     */
    public function boot()
    {
        Carbon::setLocale('pt_BR');
        JsonResource::withoutWrapping();
        Aliquota::observe(AliquotaObserver::class);
        Categoria::observe(CategoriaObserver::class);
        Customer::observe(CustomerObserver::class);
        FormaPagamento::observe(FormaPagamentoObserver::class);
        Measure::observe(MedidaObserver::class);
        Permission::observe(PermissionObserver::class);
        PontoDelivery::observe(PontoDeliveryObserver::class);
        Product::observe(ProductObserver::class);
        ProdutoConfig::observe(ProdutoConfigObserver::class);
        Profile::observe(ProfileObserver::class);
        Sector::observe(SetorObserver::class);
        Supplier::observe(SupplierObserver::class);
        User::observe(UserObserver::class);
        Master::observe(MasterObserver::class);

        // Evitar erro na primeira execução caso migracão ainda não tenha sido executada
        if (!\Schema::hasTable('gestor_configs')) {
            return;
        }

        $configs = GestorConfig::first();
        if (!$configs) {
            return;
        }

        try {
            $configs->password = CryptoHelper::decryptPassword($configs->password);
        } catch (\Exception $e) {
            Log::error('Falha ao descriptografar a senha do Gestor [EER3A]: '. $e->getMessage());
        }

            // Agora podemos atualizar o singleton corretamente
        $this->app->extend(GestorAuthService::class, function ($service, $app) use ($configs) {
            return new GestorAuthService(
                $configs->url,
                $configs->usuario,
                $configs->password
            );
        });
    }
}
