<?php


namespace App\Services\Gestor;

use App\Entities\Financeiro\TipoPagamento;
use App\Entities\Nati\Extract;
use App\Entities\Nati\Product;
use App\Entities\Financeiro\ContasReceber;
use App\Entities\GestorConfig;


class GestorIntegrationService
{
    protected $authService;

    public function __construct(GestorAuthService $authService)
    {
        $this->authService = $authService;
    }

    private function getInvoice($invoiceId, $clientId)
{
    $extracts = Extract::select('cli_Extrato.*', 'cli_itemExtrato.*')
                        ->join('cli_itemExtrato', function ($join) {
                            $join->on('cli_Extrato.idExtrato', '=', 'cli_itemExtrato.idExtrato')
                                ->on('cli_Extrato.idAbertura', '=', 'cli_itemExtrato.idAbertura');
                        })
                        ->where('cli_Extrato.idFatura', $invoiceId)
                        ->where('cli_Extrato.idCliente', $clientId)
                        ->get();

    if ($extracts->isEmpty()) {
        throw new \Exception('Fatura não encontrada');
    }

    $invoiceTotal = 0;
    $invoiceDiscount = 0;
    $invoiceService = 0;
    $invoicePaid = 0;
    $items = [];

    foreach ($extracts as $extract) {
        $invoiceTotal += $extract->vrTotal;
        $invoiceDiscount += $extract->vrDesconto;
        $invoiceService += $extract->vrAcrescimo;
        $invoicePaid += $extract->vrQuitado;

        $product = Product::find($extract->idProduto);
        if (!$product) {
            continue;
        }

        // Ajusta a quantidade se o produto for KG
        $quantity = $extract->nrQuantidade;
        if (strtoupper($product->medida->stMedida) === 'KG') {
            $quantity = $quantity / 1_000;
        }

        // Verifica se o item já existe na lista para agrupar
        $existingKey = null;
        foreach ($items as $key => $item) {
            if ($item["productId"] === $extract->idProduto) {
                $existingKey = $key;
                break;
            }
        }

        if ($existingKey !== null) {
            // Se já existe, soma quantidade e total
            $items[$existingKey]['quantity'] += $quantity;
            $items[$existingKey]['total'] += $extract->vrTotal;
        } else {
            // Se não existe, adiciona um novo item
            $items[] = [
                "id" => $extract->idItem,
                "quantity" => $quantity,
                "price" => $extract->vrUnitario,
                "discount" => $extract->vrDesconto,
                "total" => $extract->vrTotal,
                "productId" => $extract->idProduto,
                "description" => $product->stProduto,
                "cfopEntrada" => $product->aliquota->CfopEntrada,
                "cfopSaida" => $product->aliquota->CfopSaida,
                "cstIcms" => $product->aliquota->CstIcms,
                "cstPis" => $product->aliquota->CstPis,
                "cstCofins" => $product->aliquota->CstCofins,
                "iva" => $product->aliquota->iva,
                "aliquotaIcms" => $product->aliquota->AliquotaIcms,
                "aliquotaCofins" => $product->aliquota->AliquotaCofins,
                "aliquotaPis" => $product->aliquota->AliquotaPis,
                "descricao" => $product->stProduto,
                "valor" => $product->vrUnitario,
                "valor_custo" => $product->vrCusto,
                "ncm" => $product->nrNCM,
                "cest" => $product->stCEST,
                "unidade" => $product->medida->stMedida,
                "codigo" => $product->codigos[0]->idCodigo ?? null,
                "categoria" => $product->categoria->stCategoria ?? null,
                "subcategoria" => $product->subCategoria->stSubCategoria ?? null,
            ];
        }
    }

    $firstExtract = $extracts->first();
    $client = $firstExtract->cliente;

    $invoice = [
        "total" => $invoiceTotal,
        "discount" => $invoiceDiscount,
        "service" => $invoiceService,
        "paid" => $invoicePaid,
        "itens" => $items,
        "client" => [
            "id" => $client->idCliente ?? null,
            "name" => $client->stCliente ?? null,
            "shortName" => $client->stNomeFantasia ?? null,
            "email" => $client->email ?? null,
            "phone" => $client->telefone ?? null,
            "doc" => $client->stCPF_CNPJ ?? null,
            "ie_rg" => $client->stRG_IE ?? null,
            "address" => [
                "zip" => optional($client->endereco)->stCEP,
                "street" => optional($client->endereco)->stEndereco,
                "number" => $client->nrNumero ?? null,
                "complement" => $client->stComplemento ?? null,
                "neighborhood" => optional($client->endereco)->stBairro,
                "city" => optional($client->endereco)->stCidade,
                "state" => optional($client->endereco)->stEstado,
            ]
        ]
    ];

    return $invoice;
}


    public function sendInvoice($request)
    {
        $configs = GestorConfig::first();

        if(!$configs->url) {
            throw new \Exception('Url do gestor não encontrada');
        }

        $token = $this->authService->getToken();

        $invoice = $this->getInvoice($request['invoiceId'], $request['clientId']);
        $client = new \GuzzleHttp\Client();
        $response = $client->post($configs->url . '/nati/integration/venda', [
            'headers' => [
                'Authorization' => 'Bearer ' . $token,
                'nati-token' => $configs->token,
                'Accept' => 'application/json',
            ],
            'json' => $invoice,
        ]);
        $response = json_decode($response->getBody()->getContents(), true);
        $this->setExternalCode($request['invoiceId'], $request['clientId'], $response['venda_id']);
        return $response;
    }

    public function setExternalCode($invoiceId, $clientId, $externalCode)
    {
        try {
            $invoice = ContasReceber::where('idContasReceber', $invoiceId)
                                    ->where('idCliente', $clientId)
                                    ->first();
            if (!$invoice) {
                throw new \Exception("Fatura não encontrada para idFatura: $invoiceId e idCliente: $clientId");
            }

            ContasReceber::where('idContasReceber', $invoiceId)
                     ->where('idCliente', $clientId)
                     ->update(['externalCode' => $externalCode]);
        } catch (\Exception $e) {
            throw new \Exception('Falha ao salvar external code da fatura: ' . $e->getMessage());
        }

    }
}
