<?php

namespace App\Http\Controllers\Delivery;

use App\Http\Requests\Delivery\IfoodOrderCreateRequest;
use App\Services\Delivery\MobyoIntegrationService;
use App\Services\Delivery\PrintService;
use App\Services\Delivery\SatService;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Routing\Controller;
use Prettus\Validator\Exceptions\ValidatorException;

class MobyoIntegrationController extends Controller
{
    protected $mobyoIntegrationService;
    protected $printService;
    protected $satService;

    public function __construct(MobyoIntegrationService $service, PrintService $printService, SatService $satService)
    {
        $this->printService = $printService;
        $this->mobyoIntegrationService = $service;
        $this->satService = $satService;
    }

    /**
     * Display a listing of the resource.
     * @return \Illuminate\Http\JsonResponse
     */
    public function index()
    {
        return response()->json([], 200);
    }


    /**
     * Store a newly created resource in storage.
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(IfoodOrderCreateRequest $request)
    {
        try {
            if(strtoupper($request['engineType']) == 'NATI') {
                return;
            }
            return response()->json($this->mobyoIntegrationService->save($request->all()));
        } catch (ValidatorException $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessageBag()
            ], 400);
        } catch (\Exception $exception) {
            $this->generateLogError($request->all(), $exception);
            return response()->json([
                'error' => true,
                'message' => $exception->getMessage()
            ], 400);
        }
    }

    public function generateLogError($data, $exception)
    {
        $path = $this->generatePath($data);
        $messageError = $exception->getMessage();
        $log = 'Order: ' . json_encode($data) . PHP_EOL . PHP_EOL . 'Error: ' . $messageError;

        $fopen = @fopen($path, 'x+');
        @fwrite($fopen, $log);
        @fclose($fopen);
    }

    private function generatePath($data)
    {
        if(!is_dir(base_path() . '\\mobyo_errors\\')) {
            mkdir(base_path() . '\\mobyo_errors\\', 0700);
        }
        $nameFileLog = $this->generateNameFileLog($data);
        $path = base_path() . '\\mobyo_errors\\' .  $nameFileLog;
        return $path;
    }

    private function generateNameFileLog($data)
    {
        $today = getdate();
        $dateNow = $today['year'] . '-' . str_pad($today['mon'], 2, '0', STR_PAD_LEFT) . '-' . str_pad($today['mday'], 2, '0', STR_PAD_LEFT) . '_' . str_pad($today['hours'], 2, '0', STR_PAD_LEFT) . str_pad($today['minutes'], 2, '0', STR_PAD_LEFT) . str_pad($today['seconds'], 2, '0', STR_PAD_LEFT);
        $referenceOrder = str_replace('#', '', $data['reference']);
        $nameFileLog = $referenceOrder . '_' . $dateNow . '.txt';
        return $nameFileLog;
    }

    /**
     * Remove the specified resource from storage.
     * @param string $idAtendimento
     */
    public function printCoupon($idAtendimento)
    {
        try {
            $this->printService->printCoupon($idAtendimento);
            return response()->noContent();
        } catch (\Exception $exception) {
            return response()->json([
                'error' => true,
                'message' => $exception->getMessage()
            ], 400);
        }
    }

    /**
     * Remove the specified resource from storage.
     * @param string $idAtendimento
     */
    public function printRemoteCoupon($idAtendimento)
    {
        try {
            $this->printService->printRemoteCoupon($idAtendimento);
            return response()->noContent();
        } catch (\Exception $exception) {
            return response()->json([
                'error' => true,
                'message' => $exception->getMessage()
            ], 400);
        }
    }
}
