<?php

namespace App\Http\Controllers;

use Exception;
use App\Entities\User;
use App\Entities\Logs;
use App\Entities\Manager;
use Illuminate\Http\Request;
use App\Services\ManagerService;
use Illuminate\Http\JsonResponse;
use Illuminate\Routing\Controller;

class ManagerController extends Controller
{
    private $service;

    public function __construct(ManagerService $service)
    {
        $this->service = $service;
    }

    /**
     * Display a listing of the resource.
     * @return JsonResponse
     */
    public function index(Request $request)
    {
        try {
            $res = $this->service->fetch($request->all());
            return response()->json($res);
        } catch (Exception $exception) {
            return response()->json([
                'Error: ' => true,
                'Message' => $exception->getMessage()
            ], 404);
        }
    }

    /**
     * Show the specified resource.
     * @param string $id
     * @return JsonResponse
     */
    public function show(Request $request, $id)
    {
        try {
            $res = $this->service->show($request->all(), $id);
            return response()->json($res);
        } catch (Exception $exception) {
            return response()->json([
                'Error: ' => true,
                'Message' => $exception->getMessage()
            ], 404);
        }
    }

    public function sangrias($id)
    {
        return response()->json($this->service->getDetails($id, 'SANGRIA'));
    }

    public function suprimentos($id)
    {
        return response()->json($this->service->getDetails($id, 'SUPRIMENTOS'));
    }

    public function despesas($id)
    {
        return response()->json($this->service->getDetails($id, 'DESPESA'));
    }

    public function vendasPendura($id)
    {
        return response()->json($this->service->getDetails($id, 'VENDA CONTA ASSINADA'));
    }

    public function depositos($id)
    {
        return response()->json($this->service->getDetails($id, 'DEPOSITOS'));
    }

    public function vendas($id)
    {
        return response()->json($this->service->getDetails($id, 'VENDAS'));
    }

    public function payments($idAbertura, $idFormaPagamento)
    {
        return response()->json($this->service->getPayments($idAbertura, $idFormaPagamento));
    }


    /**
     * Show the form for editing the specified resource.
     * @param Request $request
     * @param string $id
     * @return JsonResponse
     */
    public function details(Request $request, string $id)
    {
        try {
            return response()->json($this->service->getDetails($id, $request->get('action')));
        } catch (Exception $exception) {
            return response()->json([
                'Error: ' => true,
                'Message' => $exception->getMessage()
            ], 404);
        }
    }

    public function getDetalhesVenda($idAbertura, $idVenda)
    {
        try {
            return response()->json($this->service->getDetalhesVenda($idAbertura, $idVenda));
        } catch (Exception $exception) {
            return response()->json([
                'Error: ' => true,
                'Message' => $exception->getMessage()
            ], 404);
        }
    }

    /*
     * Show the conferred managers
     * @param $request
     * @return JsonResponse
     * */
    public function getAllClosed(Request $request)
    {
        try {
            return response()->json($this->service->getAllClosed($request->all()));
        } catch (Exception $exception) {
            return response()->json([
                'Error: ' => true,
                'Message' => $exception->getMessage()
            ], 404);
        }
    }

    public function getDadosCaixa($idAbertura)
    {
        $gerenciador = Manager::find($idAbertura);
        $data = [];
        $data['totalClientes'] = $gerenciador->QuantityTotal();


        return response()->json($data);
    }

    public function close(Request $request, $id)
    {
        try {
            $user = User::find($request['idOperador']);
            $response = $this->service->closedPdv($request->all(), $id);
            Logs::writeLog($user->stApelido, "FECHAMENTO", 'FECHAMENTO DO CAIXA ['. $id .'] REALIZADO.');
            return response()->json(['Success' => true, 'Message' => 'Caixa fechado com sucesso']);
        } catch (Exception $exception) {
            return response()->json([
                'Error: ' => true,
                'Message' => $exception->getMessage()
            ], 500);
        }
    }

    public function getNumbersPDV(Request $request)
    {
        try{
            $pdvs = $this->service->getNumbersPDV();
            return response()->json($pdvs);
        } catch (Exception $exception) {
            return response()->json([
                'Error: ' => true,
                'Message' => $exception->getMessage()
            ], 500);
        }
    }
    public function reOpen(Request $request, $idAbertura)
    {
        try{
            $user = auth()->user();
            $this->service->reOpen($idAbertura);
            Logs::writeLog($user->stApelido, "FECHAMENTO", 'ABERTURA: ['. $idAbertura .'] REABERTA.');
            return response()->json(['Success' => true, 'Message' => 'Reabertura realizada com sucesso']);
        } catch (Exception $exception) {
            return response()->json([
                'Error: ' => true,
                'Message' => $exception->getMessage()
            ], 500);
        }
    }
    public function getAvailableMethodsPayments(Request $request)
    {
        try {
            return response()->json($this->service->getAvailableMethodsPayments());
        } catch (Exception $exception) {
            return response()->json([
                'Error: ' => true,
                'Message' => $exception->getMessage()
            ], 500);
        }

    }
    public function updatePaymentMethod(Request $request)
    {
        try{
            $user = auth()->user();
            $response = $this->service->updatePaymentMethod($request->all());
            Logs::writeLog($user->stApelido, "FECHAMENTO", 'FORMA DE PAGAMENTO DA VENDA ['. $request["idAbertura"] .' \ '. $request["idVenda"] .'] ALTERADA.');
            return response()->json(['Success' => true, 'Message' => 'Forma de pagamento alterada com sucesso', 'data' => $response[0]]);
        } catch (Exception $e) {
            return response()->json([
                'Error: ' => true,
                'Message' => $e->getMessage()
            ], 500);
        }

    }
    public function getCancelamentos(Request $request, $idAbertura)
    {
        try{
            return response()->json($response = $this->service->getCancelamentos($idAbertura));
        } catch (Exception $e) {
            return response()->json([
                'Error: ' => true,
                'Message' => $e->getMessage()
            ], 500);
        }
    }
    public function delete(Request $request, $id)
    {
        try{
            $user = auth()->user();
            $response = $this->service->delete($id);
            Logs::writeLog($user->stApelido, "FECHAMENTO", 'ABERTURA ['. $id .'] EXCLUIDA.');
            return response()->json(['Success' => true, 'Message' => 'Abertura excluida com sucesso']);
        } catch (Exception $e) {
            return response()->json([
                'Error: ' => true,
                'Message' => $e->getMessage()
            ], 500);
        }
    }
    public function getEstornos($idAbertura)
    {
        try{
            return response()->json($response = $this->service->getEstornos($idAbertura));
        } catch (Exception $e) {
            return response()->json([
                'Error: ' => true,
                'Message' => $e->getMessage()
            ], 500);
        }
    }
}
