<?php

namespace App\Http\Controllers\Nati;

use Illuminate\Http\Request;

use App\Http\Requests;
use Prettus\Validator\Contracts\ValidatorInterface;
use Prettus\Validator\Exceptions\ValidatorException;
use App\Http\Requests\RespostaProdutoCreateRequest;
use App\Http\Requests\RespostaProdutoUpdateRequest;
use App\Repositories\Nati\RespostaProdutoRepository;
use App\Validators\Nati\RespostaProdutoValidator;

/**
 * Class RespostaProdutosController.
 *
 * @package namespace App\Http\Controllers\Nati;
 */
class RespostaProdutosController extends Controller
{
    /**
     * @var RespostaProdutoRepository
     */
    protected $repository;

    /**
     * @var RespostaProdutoValidator
     */
    protected $validator;

    /**
     * RespostaProdutosController constructor.
     *
     * @param RespostaProdutoRepository $repository
     * @param RespostaProdutoValidator $validator
     */
    public function __construct(RespostaProdutoRepository $repository, RespostaProdutoValidator $validator)
    {
        $this->repository = $repository;
        $this->validator  = $validator;
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $this->repository->pushCriteria(app('Prettus\Repository\Criteria\RequestCriteria'));
        $respostaProdutos = $this->repository->all();

        if (request()->wantsJson()) {

            return response()->json([
                'data' => $respostaProdutos,
            ]);
        }

        return view('respostaProdutos.index', compact('respostaProdutos'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  RespostaProdutoCreateRequest $request
     *
     * @return \Illuminate\Http\Response
     *
     * @throws \Prettus\Validator\Exceptions\ValidatorException
     */
    public function store(RespostaProdutoCreateRequest $request)
    {
        try {

            $this->validator->with($request->all())->passesOrFail(ValidatorInterface::RULE_CREATE);

            $respostaProduto = $this->repository->create($request->all());

            $response = [
                'message' => 'RespostaProduto created.',
                'data'    => $respostaProduto->toArray(),
            ];

            if ($request->wantsJson()) {

                return response()->json($response);
            }

            return redirect()->back()->with('message', $response['message']);
        } catch (ValidatorException $e) {
            if ($request->wantsJson()) {
                return response()->json([
                    'error'   => true,
                    'message' => $e->getMessageBag()
                ]);
            }

            return redirect()->back()->withErrors($e->getMessageBag())->withInput();
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int $id
     *
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $respostaProduto = $this->repository->find($id);

        if (request()->wantsJson()) {

            return response()->json([
                'data' => $respostaProduto,
            ]);
        }

        return view('respostaProdutos.show', compact('respostaProduto'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int $id
     *
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $respostaProduto = $this->repository->find($id);

        return view('respostaProdutos.edit', compact('respostaProduto'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  RespostaProdutoUpdateRequest $request
     * @param  string            $id
     *
     * @return Response
     *
     * @throws \Prettus\Validator\Exceptions\ValidatorException
     */
    public function update(RespostaProdutoUpdateRequest $request, $id)
    {
        try {

            $this->validator->with($request->all())->passesOrFail(ValidatorInterface::RULE_UPDATE);

            $respostaProduto = $this->repository->update($request->all(), $id);

            $response = [
                'message' => 'RespostaProduto updated.',
                'data'    => $respostaProduto->toArray(),
            ];

            if ($request->wantsJson()) {

                return response()->json($response);
            }

            return redirect()->back()->with('message', $response['message']);
        } catch (ValidatorException $e) {

            if ($request->wantsJson()) {

                return response()->json([
                    'error'   => true,
                    'message' => $e->getMessageBag()
                ]);
            }

            return redirect()->back()->withErrors($e->getMessageBag())->withInput();
        }
    }


    /**
     * Remove the specified resource from storage.
     *
     * @param  int $id
     *
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $deleted = $this->repository->delete($id);

        if (request()->wantsJson()) {

            return response()->json([
                'message' => 'RespostaProduto deleted.',
                'deleted' => $deleted,
            ]);
        }

        return redirect()->back()->with('message', 'RespostaProduto deleted.');
    }
}
