<?php

namespace App\Http\Controllers;

use App\Entities\Logs;
use App\Entities\Manager;
use App\Events\NewOrdersToPreparation;
use App\Services\UsersService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use App\Http\Requests\UserCreateRequest;
use App\Http\Requests\UserUpdateRequest;
use App\Http\Resources\User as UserResource;
use App\Entities\User;

/**
 * Class UsersController.
 *
 * @package namespace App\Http\Controllers;
 */
class UsersController extends Controller
{

    protected $service;
    protected $user;

    public function __construct(UsersService $service)
    {
        $this->service = $service;
    }

    public function index(Request $request)
    {
        try {
            return response()->json($this->service->get($request->query()));

        } catch (\Exception $e) {

            return response()->json([
                'error' => true,
                'message' => $e->getMessage()
            ], 404);
        }
    }

    public function nextId()
    {
        try {
            return response()->json($this->service->getNextId());

        } catch (\Exception $e) {

            return response()->json([
                'error' => true,
                'message' => $e->getMessage()
            ], 404);
        }
    }

    public function changePwd($id, $data)
    {
        try {
            return response()->json($this->service->changePwd($id, $data));
        } catch (\Exception $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessage()
            ], 404);
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param UserCreateRequest $request
     *
     * @return JsonResponse
     *
     */
    public function store(UserCreateRequest $request)
    {
        try {
            $user = auth()->user();
            $response = $this->service->create($request->all());
            Logs::writeLog($user->stApelido, "NATI", 'O OPERADOR [' . $user->stApelido . '] CADASTROU O OPERADOR [' . $response->stApelido . ']');
            return response()->json($response);

        } catch (\Exception $e) {

            return response()->json([
                'error' => true,
                'message' => $e->getMessage()
            ], 400);
        }
    }

    public function me()
    {
        $user = auth()->user();
        $data = [
            'user' => $user,
            'rules' => $user->rules(),
            'access_rules' => $user->access_rules()
        ];
        return response()->json($data);
    }

    /**
     * Display the specified resource.
     *
     * @param int $id
     *
     * @return JsonResponse
     */
    public function show($id)
    {
        try {
            $user = $this->service->getById($id);
            return response()->json(new UserResource($user));

        } catch (\Exception $e) {

            return response()->json([
                'error' => true,
                'message' => $e->getMessage()
            ], 404);
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * @param UserUpdateRequest $request
     * @param string $id
     *
     * @return JsonResponse
     *
     */
    public function update(UserUpdateRequest $request, $id)
    {
        try {
            $user = auth()->user();
            $response = $this->service->update($request->all(), $id);
            Logs::writeLog($user->stApelido, "NATI", 'O OPERADOR [' . $user->stApelido . '] EDITOU O OPERADOR [' . $response->stApelido . ']');

            return response()->json($response);

        } catch (\Exception $e) {

            return response()->json([
                'error' => true,
                'message' => $e->getMessage()
            ], 404);
        }
    }


    /**
     * Remove the specified resource from storage.
     *
     * @param int $id
     *
     * @return JsonResponse
     */
    public function destroy($id)
    {
        try {
            $user = auth()->user();
            $gerenciador = Manager::where('idOperador', $id)->where('flAberto', true)->get()->count();
            if($gerenciador > 0 ){
                throw new \Exception("Existe um caixa aberto por este usuário");
            }
            $response = $this->service->delete($id);
            Logs::writeLog($user->stApelido, "NATI", 'O OPERADOR [' . $user->stApelido . '] DESATIVOU O OPERADOR [' . $response->stApelido . ']');
            return response()->json($response, 200);

        }
        catch (\Exception $e) {

            return response()->json([
                'error' => true,
                'message' => $e->getMessage()
            ], 400);
        }
    }

    public function storeCard($idUser, $codCard)
    {
        try {
            $user = auth()->user();
            $response = $this->service->storeCard($idUser, $codCard);
            Logs::writeLog($user->stApelido, "USUARIOS", 'O OPERADOR [' . $user->idOperador . '-' . $user->stApelido . '] CADASTROU UM CARTÃO PARA O OPERADOR [' . $idUser . '-' . $response->stApelido . ']');
            return response()->json($response);
        } catch (\Exception $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessage()
            ], 404);
        }
    }

    public function getCashiers()
    {
        try {
            $cashiers = $this->service->getCashiers();
            return response()->json($cashiers);
        } catch (\Exception $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessage()
            ], 404);
        }
    }

    public function find($text)
    {
        if (strlen($text) > 0) {
            $text = '%' . $text . '%';
            $res = User::where('stApelido', 'like', $text)->get();

            if (count($res)) {
                return json_encode($res);

            } else {
                return null;
            }

        }
    }
}
