<?php

namespace App\Services\Delivery;

use App\Entities\Delivery\Item;
use App\Entities\Nati\Product;
use App\Validators\Delivery\IfoodOrderValidator;
use Carbon\Carbon;
use App\Entities\Delivery\Order;
use App\Repositories\DlvOrderRepository;
use App\Validators\Delivery\OrderValidator;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Prettus\Validator\Contracts\ValidatorInterface;
use App\Entities\Nati\FormaPagamento;
use App\Entities\Delivery\Payment;

class OrderService
{
    protected $repository;
    protected $cepService;
    protected $painelPreparoService;
    protected $customerService;
    protected $printService;
    protected $orderValidator;
    protected $ifoodOrderValidator;
    protected $config;


    public function __construct(
        DlvOrderRepository $repository,
        CustomerService $customerService,
        CepService $cepService,
        PrintService $printService,
        OrderValidator $orderValidator,
        IfoodOrderValidator $ifoodOrderValidator,
        PainelPreparoService $preparoService)
    {
        $this->repository = $repository;
        $this->cepService = $cepService;
        $this->painelPreparoService = $preparoService;
        $this->customerService = $customerService;
        $this->printService = $printService;
        $this->orderValidator = $orderValidator;
        $this->ifoodOrderValidator = $ifoodOrderValidator;
        $this->config = (object)app('config')->get('ifood');
    }

    public function getAll(){
        return $this->repository->findWhere(['stStatus'=> 'P'])->take(10); //TODO filtra por status
    }

    public function getPendingOrders()
    {
        return Order::selectRaw('dlv_Atendimentos.idAtendimento, dlv_Atendimentos.dtAbertura, dlv_Atendimentos.dtSaidaMotoboy, dlv_Atendimentos.stStatus, dlv_Atendimentos.idCliente, SUM(dlv_Itens.vrTotal) as vrTotal, dlv_Atendimentos.desconto, (sum(dlv_itens.vrTotal) - isnull(dlv_Atendimentos.desconto, 0)) as total, dlv_Clientes.stCliente, dlv_CEPs.stEndereco')
            ->where(function ($query) {
                $query->where('stStatus', 'S')
                    ->orWhere('stStatus', 'P');
            })
            ->where('dlv_Atendimentos.flAtivo', true)
            ->join('dlv_Itens', 'dlv_Itens.idAtendimento', '=', 'dlv_Atendimentos.idAtendimento')
            ->join('dlv_Clientes', 'dlv_Atendimentos.idCliente', '=', 'dlv_Clientes.idCliente')
            ->join('dlv_CEPs', 'dlv_Clientes.idCep', '=', 'dlv_CEPs.idCep')
            ->groupBy('dlv_Atendimentos.idAtendimento', 'dlv_Atendimentos.dtAbertura', 'dlv_Atendimentos.dtSaidaMotoboy', 'dlv_Atendimentos.stStatus', 'dlv_Atendimentos.idCliente', 'dlv_Clientes.stCliente', 'dlv_CEPs.stEndereco', 'dlv_Atendimentos.desconto')
            ->orderBy('dlv_Atendimentos.dtAbertura', 'DESC')
            ->limit(50)
            ->get();
    }

    public function create($data)
    {
        $data = $this->OrderConvert($data);

        $this->orderValidator->with($data)->passesOrFail(ValidatorInterface::RULE_CREATE);

        return DB::transaction(function () use ($data) {
            $order = Order::create($data);

            foreach ($data['items'] as $i) {
                $item = Item::create($i);
                $order->items()->save($item);
            }

            foreach ($data['payments'] as $p) {
                $payment = Payment::create($p);
                $order->payments()->save($payment);
            }

            return $order;
        });
    }

    public function get($id)
    {
        return Order::with(['payments', 'items', 'customer'])->find($id);
    }

    private function trataString($string)
    {
        $comAcentos = array('à', 'á', 'â', 'ã', 'ä', 'å', 'ç', 'è', 'é', 'ê', 'ë', 'ì', 'í', 'î', 'ï', 'ñ', 'ò', 'ó', 'ô', 'õ', 'ö', 'ù', 'ü', 'ú', 'ÿ', 'À', 'Á', 'Â', 'Ã', 'Ä', 'Å', 'Ç', 'È', 'É', 'Ê', 'Ë', 'Ì', 'Í', 'Î', 'Ï', 'Ñ', 'Ò', 'Ó', 'Ô', 'Õ', 'Ö', 'O', 'Ù', 'Ü', 'Ú');
        $semAcentos = array('a', 'a', 'a', 'a', 'a', 'a', 'c', 'e', 'e', 'e', 'e', 'i', 'i', 'i', 'i', 'n', 'o', 'o', 'o', 'o', 'o', 'u', 'u', 'u', 'y', 'A', 'A', 'A', 'A', 'A', 'A', 'C', 'E', 'E', 'E', 'E', 'I', 'I', 'I', 'I', 'N', 'O', 'O', 'O', 'O', 'O', '0', 'U', 'U', 'U');

        return str_replace($comAcentos, $semAcentos, $string);
    }

    public function convert($data)
    {
        $this->ifoodOrderValidator->with($data)->passesOrFail(ValidatorInterface::RULE_CREATE);

        //$pedido = Pedido::create($data);

        $exist = $this->repository->findByField('reference', $data['reference'])->first();
        if ($exist) {
            return $exist;
        }
        try {
            return DB::transaction(function () use ($data) {

                $customer = $this->getCustomer($data);//
                $dt = Carbon::parse($data['createdAt'])->timezone('America/Sao_Paulo');
                $obs = '';
                $complemento = isset($data['deliveryAddress']['complement']) ? $this->trataString($data['deliveryAddress']['complement']) : '';
                $reference = isset($data['deliveryAddress']['reference']) ? $this->trataString($data['deliveryAddress']['reference']) . chr(10) : '';
                $telefone = isset($data['customer']['phone']) ? $data['customer']['phone'] : '';
                $nome = isset($data['customer']['name'])? $data['customer']['name']: '';
                $referencia = isset($data['shortReference'])? $data['shortReference'] : '';
                $loc = isset($data['localizer']['id']) ? $data['localizer']['id'] : '';
                $entrega = isset($data['deliveryDateTime'])? Carbon::parse($data['deliveryDateTime'])->timezone('America/Sao_Paulo'): '';
                $pedidos = isset($data['customer']['ordersCountOnRestaurant']) ? $data['customer']['ordersCountOnRestaurant'] : 1;
                $enderecoEntrega = isset($data['deliveryAddress']['formattedAddress']) ? $this->trataString($data['deliveryAddress']['formattedAddress']) : '';

                if ($this->config->salvar_cliente_em_observacao) {
                    $obs = 'CLIENTE: ' . $nome . chr(10)
                        . 'TELEFONE: ' . $telefone . chr(10)
                        . 'PEDIDO: #' . $referencia . chr(10)
                        . 'FEITO AS ' . $dt->hour . ':' . $dt->minute . chr(10)
                        . 'LOCALIZADOR DO PEDIDO ' . $loc. chr(10)
                        . 'ENTREGA PREVISTA: ' . $entrega . chr(10)
                        . $pedidos . ' PEDIDOS NA SUA LOJA' . chr(10)
                        . 'ENTREGAR EM: ' . $enderecoEntrega . $this->trataString($data['deliveryAddress']['neighborhood']) . chr(10)
                        . $this->trataString((isset($data['deliveryAddress']['city']) ? $data['deliveryAddress']['city'] : '')). ' - CEP: ' . $data['deliveryAddress']['postalCode'] . chr(10)
                        . $reference
                        . $complemento . chr(10);
                }

                $cliente_retira = false;
                if (isset($data['deliveryMethod']) && $data['deliveryMethod']['mode'] == 'TAKEOUT') {
                    $cliente_retira = true;
                }

                $desconto = 0;
                if (isset($data['benefits'])) {
                    $benfits = $this->getBenefits($data['benefits'], 'CART');
                    foreach ($benfits as $be) {
                        $desconto += $be['value'];
                    }
                }

                $cpf = isset($data['customer']['taxPayerIdentificationNumber'])
                    ? str_replace(['.', '-','/'], '', $data['customer']['taxPayerIdentificationNumber'])
                    : null;
                $cliente = isset($data['customer']['name']) ? Str::upper($data['customer']['name']) : '';

                $order = Order::create([
                    'idAtendimento' => $this->getNextId(),
                    'idCliente' => $customer,
                    'idOperador' => 999,
                    'dtAbertura' => Carbon::parse($data['createdAt'])->timezone('America/Sao_Paulo'),
                    'dtConclusao' => Carbon::parse($data['createdAt'])->timezone('America/Sao_Paulo'),
                    'stStatus' => 'P',
                    'stOBS' => $obs,
                    'reference' => $data['reference'],
                    'flPago' => false,
                    'flClienteRetira' => $cliente_retira,
                    'flAgendarPedido' => $data['scheduled'],
                    'dtAgendarPedido' => isset($data['preparationStartDateTime']) ? Carbon::parse($data['preparationStartDateTime'])->timezone('America/Sao_Paulo') : '',
                    'dtAgendarPedidoProducao' => isset($data['preparationStartDateTime']) ? Carbon::parse($data['preparationStartDateTime'])->timezone('America/Sao_Paulo') : '',
                    'desconto' => $desconto,
                    'cliente' => $cliente,
                    'stCpf' => $cpf,
                    'engine' => 'IFOOD'
                ]);

                $count = 1;

                foreach ($data['items'] as $it) {
                    if (isset($it['externalCode'])) {

                        $produto = $this->getIdProdutoByCode($it['externalCode']);
                        if (!$produto) {
                            continue;
                        }

                        if ($produto['idTipoProduto'] == '3' && isset($it['subItems'])) {
                            $frac = $this->filterFracionado($it['subItems']);
                            $normal = $this->filterNormal($it['subItems']);

                            if ($frac) {

                                $observacoes = isset($it['observations']) ? $this->trataString($it['observations']) : '';
                                $produto = $this->getIdProdutoByCode($frac[1]['externalCode']);
                                $item = $this->makeItem($frac[1], $produto, $it['quantity'], $order->idAtendimento, $count);
                                $item->stOBS = $observacoes;
                                $total = 0;
                                if (count($frac) == 1) {
                                    $produto = $this->getIdProdutoByCode($frac[1]['externalCode']);
                                    $item->idProduto = $produto['idProduto'];
                                    $item->nrQuantidade = $frac[1]['quantity'];
                                } else {
                                    foreach ($frac as $i) {
                                        $produto = $this->getIdProdutoByCode($i['externalCode']);
                                        if ($produto) {
                                            $total += $i['totalPrice'];
                                            $item->stCodigos .= (strlen($item->stCodigos) >= 1 ? "@" : "") . $produto['idProduto'];
                                            $item->stValores .= (strlen($item->stValores) >= 1 ? "@" : "") . $i['totalPrice'];
                                            $item->stQuantidades .= (strlen($item->stQuantidades) >= 1 ? "@" : "") . $i['quantity'] . '/' . count($frac);
                                        }
                                    }
                                    $item->vrUnitario = $total;
                                    $item->vrTotal = $total;
                                }

                                $order->items()->save($item);
                                $count++;
                            }
                            if ($normal) {
                                foreach ($normal as $i) {
                                    if (isset($i['externalCode'])) {
                                        $produto = $this->getIdProdutoByCode($i['externalCode']);
                                        if ($produto) {
                                            $item = $this->makeItem($i, $produto, 1, $order->idAtendimento, $count);
                                            $order->items()->save($item);
                                            $count++;
                                        }
                                    }
                                }
                            }
                        } else {
                            $item = $this->makeItem($it, $produto, 1, $order->idAtendimento, $count);
                            if ($item) {
                                $order->items()->save($item);
                                $count++;
                            }
                            if (isset($it['subItems'])) {
                                foreach ($it['subItems'] as $i) {
                                    $produto = $this->getIdProdutoByCode($i['externalCode']);
                                    if ($produto) {
                                        $item = $this->makeItem($i, $produto, $it['quantity'], $order->idAtendimento, $count);
                                        if ($item) {
                                            $order->items()->save($item);
                                            $count++;
                                        }
                                    }
                                }
                            }
                        }
                    } else {
                        if (isset($it['subItems'])) {
                            foreach ($it['subItems'] as $i) {
                                $produto = $this->getIdProdutoByCode($i['externalCode']);
                                if ($produto) {
                                    $item = $this->makeItem($i, $produto, $it['quantity'], $order->idAtendimento, $count);
                                    if ($item) {
                                        $order->items()->save($item);
                                        $count++;
                                    }
                                }
                            }
                        }
                    }
                }

                // taxa de entrega
                if ($this->config->salvar_taxa_entrega) {
                    if (isset($data['deliveryFee'])) {
                        $produto = $this->getIdProdutoByCode('-100');
                        if ($produto) {
                            $item = new Item([
                                'idAtendimento' => $order->idAtendimento,
                                'nrItem' => $count,
                                'idProduto' => $produto['idProduto'],
                                'stOperador' => 'INTEGRACAO',
                                'nrQuantidade' => 1,
                                'dtData' => Carbon::now(),
                                'vrUnitario' => $data['deliveryFee'],
                                'vrTotal' => $data['deliveryFee'],
                                'stOBS' => "",
                                'stObservacoes' => "",
                                'stValores' => "",
                                'stCodigos' => "",
                                'stQuantidades' => "",
                                'flPrint' => true,
                                'idComanda' => "",
                                'dtAlteracao' => Carbon::now(),
                                'flAtivo' => true,
                            ]);
                        }

                        $order->items()->save($item);
                    }
                }
                // Total
                $total = 0;
                foreach ($order->items()->get() as $item){
                    $total += $item['vrTotal'];
                }

                // formas de pagamento
                foreach ($data['payments'] as $pay) {
                    $vrPagamento = 0;
                    $entrega = 0;
                    if ($this->config->salvar_taxa_entrega) {
                        $benfits = [];
                        $beneficio_entrega = 0;
                        if($this->config->salvar_taxa_entrega){
                            $entrega = $data['deliveryFee'];
                        }

                        if (isset($data['benefits'])) {
                            $benfits = $this->getBenefits($data['benefits'], 'DELIVERY_FEE');
                            if (count($benfits) > 0) {
                                foreach ($benfits as $b) {
                                    $beneficio_entrega += $b['value'];
                                }
                            }
                        }

                        $vrPagamento = $total - ($entrega - $beneficio_entrega);

                    } else {
                        $vrPagamento = $total;
                    }
                    $idFormaPagamento = 0;
                    if ($this->config->forma_pagamento_fixa) {
                        $idFormaPagamento = $this->config->id_forma_pagamento;
                    } else {
                        $idFormaPagamento = $this->getIdFormaPagamento($pay['code']);
                    }
                    $payment = new Payment();
                    $payment->idFicha = $this->getNextPaymentId($order->idAtendimento);
                    $payment->idAtendimento = $order->idAtendimento;
                    $payment->idFormaPagamento = $idFormaPagamento;
                    $payment->vrFormaPagamento = $vrPagamento;
                    $payment->dtData = Carbon::parse($data['createdAt'])->timezone('America/Sao_Paulo');
                    $payment->flTroco = false;
                    $payment->flAtivo = true;
                    $payment->dtAlteracao = Carbon::parse($data['createdAt'])->timezone('America/Sao_Paulo');
                    $order->payments()->save($payment);
                }
                try {
                    // impressão ficha tecnica
                    if ($this->config->emite_cupom_remoto) {
                        $this->printService->printRemoteCoupon($order->idAtendimento);
                    }
                    // impressão cupom delivery
                    if ($this->config->emite_cupom_delivery) {
                        $this->printService->printCoupon($order->idAtendimento);
                    }
                    if ($this->config->usar_painel_preparo) {
                        $this->painelPreparoService->add($order);
                    }
                } catch (\Exception $e) {
                }
                return $order;
            });
        } catch (\Exception $e) {
            throw new \Exception($e);
        }
    }

    private function getBenefits($data, $target = '')
    {
        try {
            return array_filter($data, function ($it) use ($target) {
                return $it['target'] == $target;
            });
        } catch (\Exception $e) {
            return [];
        }
    }

    private function filterFracionado(array $data)
    {
        $res = array_filter($data, function ($it) {
            if (isset($it['externalCode'])) {
                $produto = $this->getIdProdutoByCode($it['externalCode']);
                return $produto['idTipoProduto'] == '3';
            }
        });

        return $res;
    }

    private function filterNormal(array $data)
    {
        $res = array_filter($data, function ($it) {
            if (isset($it['externalCode'])) {

                $produto = $this->getIdProdutoByCode($it['externalCode']);
                return $produto['idTipoProduto'] != '3';
            }
        });

        return $res;
    }

    private function makeItem($data, $produto = null, $quantidade = 1, $idAtendimento, $count)
    {
        try {
            if ($produto) {
                $qtd = floatval($data['quantity']) * $quantidade;
                $observacoes = isset($data['observations']) ? $this->trataString($data['observations']) : '';
                $unitario = floatval($data['price']) > 0 ? floatval($data['price']) : 0;
                $total = $unitario * $qtd;
                $item = new Item([
                    'idProduto' => $produto['idProduto'],
                    'stOperador' => 'INTEGRACAO',
                    'nrQuantidade' => $qtd,
                    'dtData' => Carbon::now(),
                    'vrUnitario' => $data['price'],
                    'vrTotal' => number_format($total, 2),
                    'idAtendimento' => $idAtendimento,
                    'nrItem' => $count,
                    'stObservacoes' => '',
                    'stValores' => "",
                    'stCodigos' => "",
                    'stQuantidades' => "",
                    'flPrint' => true,
                    'idComanda' => "",
                    'dtAlteracao' => Carbon::now(),
                    'flAtivo' => true,
                    'stOBS' => $observacoes,
                    'codigo' => $produto['codigo'],
                    'descricao' => $produto['descricao'],
                    'cfop' => $produto['cfop'],
                    'ncm' => $produto['ncm'],
                    'cstIcms' => $produto['cstIcms'],
                    'aliquotaIcms' => number_format($produto['aliquotaIcms'], 2),
                    'cstPis' => $produto['cstPis'],
                    'aliquotaPis' => number_format($produto['aliquotaPis'], 2),
                    'cstCofins' => $produto['cstCofins'],
                    'aliquotaCofins' => number_format($produto['aliquotaCofins'], 2),
                    'ibpt' => number_format($produto['ibpt'] / 100 * $total, 2),
                    'medida' => $produto['medida'],
                ]);
                return $item;
            }
        } catch (\Exception $e) {
            throw new \Exception('NATI: Item nao encontrado');
        }
    }

    private function getCustomer($data)
    {
        if ($this->config->cliente['clienteFixo']) {
            try {
                $merchant = $data['merchant']['shortId'];

                $x = array_where($this->config->cliente['idClienteFixo'], function ($key, $value) use ($merchant) {
                    return $value == $merchant;
                });

                return $x[$merchant];
            } catch (\Exception $e) {
                throw  new \Exception("NATI: Cliente não configurado");
            }
        } else {
            $customer = $this->customerService->getByName($data['customer']['name']);
            if (!$customer) {
                $customer = $this->customerService->create([
                    'stCliente' => $data['name'],
                    'flPessoaJuridica' => false,
                    'nrNumero' => $data['deliveryAddress']['streetNumber'],
                    'idCEP' => $this->GetIdCep($data['deliveryAddress']),
                    'flBloqueado' => false,
                    'flAtivo' => true,
                    'dtCadastro' => Carbon::now(),
                    'stOperador' => 'IFOOD',
                    'dtUltimaAlteracao' => Carbon::now(),
                    'stOperadorUltimaAlteracao' => 'IFOOD',
                ]);
            }
            return $customer->idCliente;
        }
    }

    private function getIdProdutoByCode($data)
    {
        try {
            return Product::select([
                'prd_Produtos.idProduto',
                'prd_Codigos.idCodigo as codigo',
                'prd_produtos.stProdutoAbreviado as descricao',
                'prd_produtos.nrNcm as ncm',
                'prd_Aliquotas.cfopSaida as cfop',
                'prd_Aliquotas.CstIcms as cstIcms',
                'prd_Aliquotas.AliquotaIcms as aliquotaIcms',
                'prd_Aliquotas.CstPis as cstPis',
                'prd_Aliquotas.AliquotaPis as aliquotaPis',
                'prd_Aliquotas.CstCofins as cstCofins',
                'prd_Aliquotas.AliquotaCofins as aliquotaCofins',
                'prd_Medidas.stMedida as medida',
                'tbt_CargaTributariaMedia.vrAliqNac as ibpt',
                'prd_Produtos.idTipoProduto',
            ])
                ->join('prd_codigos', 'prd_produtos.idProduto', '=', 'prd_Codigos.idProduto')
                ->join('prd_Aliquotas', 'prd_produtos.idAliquota', '=', 'prd_Aliquotas.idAliquota')
                ->join('prd_Medidas', 'prd_produtos.idMedida', '=', 'prd_Medidas.idMedida')
                ->join('tbt_CargaTributariaMedia', 'prd_produtos.nrNcm', '=', 'tbt_CargaTributariaMedia.stCodigo')
                ->where(function ($query) use ($data) {
                    $query->where('prd_Codigos.idCodigo', '=', "$data");
                })->first();
        } catch (\Exception $e) {
            throw  new \Exception("NATI: Codigo do produto não encontrado");
        }

    }

    private function getIdFormaPagamento($code)
    {
        switch ($code) {
            case 'DIN':
                $x = FormaPagamento::where('idTipo', 1)->first()->idFormaPagamento;
                return $x;
                break;
            default:
                $x = FormaPagamento::where('idTipo', 2)->first()->idFormaPagamento;
                return $x;
                break;
        }
    }

    private function getNextPaymentId($idAbertura)
    {
        return Payment::where('idAtendimento', $idAbertura)->max('idFicha') + 1;
    }

    private function getNextId()
    {
        return Order::max('idAtendimento') + 1;
    }

    private function GetIdCep($data)
    {
        $cep = $this->cepService->getByCep($data['postalCode']);
        if ($cep) {
            return $cep;
        } else {
            $res = $this->cepService->create([
                'stCEP' => $data['postalCode'],
                'stEndereco' => $data['streetName'],
                'stBairro' => $data['neighborhood'],
                'stCidade' => $data['city'],
                'stEstado' => $data['state'],
            ]);
            return $res->idCEP;
        }
    }

    public function dispatchOrReceive($request)
    {
        DB::transaction(function () use ($request) {
            $id = "";

            for ($x = 0; $x < strlen($request['stCupomDelivery']); $x++) {
                if (ctype_digit($request['stCupomDelivery'][$x]))
                    $id .= $request['stCupomDelivery'][$x];
            }

            $order = Order::findOrFail($id);

            if (!$order->flAtivo)
                throw new \Exception("O pedido informado está excluido.");
            elseif ($order->flOrcamento)
                throw new \Exception("Você não pode enviar um orçamento para entrega.");
            else {
                switch ($order->stStatus) {
                    case 'F':
                        throw new \Exception("O pedido informado está em formulação.");
                    case 'E':
                        throw new \Exception("O pedido informado já foi entregue.");
                    case 'P':
                        if ($request['statusDr'] != 2) {
                            $order->stStatus = 'S';
                            $order->dtSaidaMotoboy = Carbon::now();
                            $order->idMotoboy = $request['idMotoboy'];
                            $order->save();
                            return [
                                'message' => "O pedido foi enviado com sucesso."
                            ];
                        } else {
                            throw new \Exception("O pedido informado ainda não foi despachado.");
                        }
                    case 'S':
                        if ($request['statusDr'] != 1) {
                            $order->stStatus = 'E';
                            $order->dtSaidaMotoboy = Carbon::now();
                            $order->save();
                            return [
                                'message' => "O pedido foi finalizado com sucesso."
                            ];
                        } else {
                            throw new \Exception("O pedido informado já foi despachado.");
                        }
                }
            }
        });
    }

    public function OrderConvert($data)
    {
        $user = auth()->user();
        $order = [
            'idAtendimento' => $this->getNextId(),
            'idCliente' => $data['customer'],
            'idOperador' => $user->idOperador,
            'dtAbertura' => Carbon::now(),
            'dtConclusao' => Carbon::now(),
            'items' => [],
            'payments' => [],
        ];

        foreach ($data['items'] as $i) {
            $item = [
                'nrItem' => $i['id'],
                'idAtendimento' => $order['idAtendimento'],
                'stOperador' => $user->stApelido,
                'idProduto' => $i['productId'],
                'nrQuantidade' => $i['quantity'],
                'vrUnitario' => $i['price'],
                'vrTotal' => $i['totalPrice'],
                'dtData' => Carbon::now(),
                'dtAlteracao' => Carbon::now(),
                'idComanda' => 0,
                'flPrint' => false,
                'flAtivo' => true,
            ];

            array_push($order['items'], $item);
        }

        $count = 1;
        foreach ($data['payments'] as $p) {
            $payment = [
                'idFicha' => $count,
                'idAtendimento' => $order['idAtendimento'],
                'idFormaPagamento' => $p['paymentId'],
                'vrFormaPagamento' => $p['value'],
                'dtData' => Carbon::now(),
                'flTroco' => $p['change'],
                'flAtivo' => true,
                'dtAlteracao' => Carbon::now()
            ];
            array_push($order['payments'], $payment);
            $count++;
        }
        return $order;
    }

    public function FindByField($field, $value)
    {
        return $this->repository->findByField($field, $value)->first();
    }
}
