<?php

namespace App\Services\Delivery;

use Exception;
use GuzzleHttp\Client;
use App\Entities\Nati\FormaPagamento;
use App\Events\DeliveryOrderCfeCreated;
use GuzzleHttp\Exception\ClientException;

class SatService
{
    protected $venda;
    protected $client;
    protected $config;
    protected $orderService;

    public function __construct(OrderService $orderService)
    {
        $this->orderService = $orderService;
        $this->config = (object)app('config')->get('fiscalManager');
        $this->venda = [
            'products' => [],
            'payments' => []
        ];
        $this->client = new Client();
    }

    /*
     * @throws Exception|\GuzzleHttp\Exception\ClientException
     * @param $idAtendimento
     */
    public function transmite($idAtendimento)
    {
        $order = $this->orderService->get($idAtendimento);
        if (!$order) {
            throw new Exception('Order not found');
        }

        if ($order->fiscalProtocolo || $order->fiscalStatus) {
            throw new Exception('Order cfe already requests');
        }

        $this->venda['pdv_id'] = 11;
        $this->venda['user_name'] = '';

        $this->venda["service"] = 0;
        $this->venda['discount'] = 0;

        if (isset($order->stCpf) && strlen($order->stCpf) == 11) {
            $this->venda['customer']['cpf'] = $order->stCpf;
        }
        if (isset($order->stCpf) && strlen($order->stCpf) == 14) {
            $this->venda['customer']['cnpj'] = $order->stCpf;
        }

        $this->makeItens($order->items()->get());
        $this->makePayment($order->payments()->get());
        try {
            $response = $this->client->post($this->config->urlPost, ['json' => $this->venda]);

            if ($response->getStatusCode() == 200) {
                $result = json_decode($response->getBody()->getContents(), true);

                $order->fiscalStatus = $result['status'];
                $order->fiscalProtocolo = $result['reference_id'];
                $order->save();
                $this->getCfe($order->fiscalProtocolo);
                return $result;
            }
        } catch (ClientException $e) {
            throw new Exception($e->getMessage());
        }
    }

    public function getCfe($id)
    {
        try {
            $response = $this->client->get($this->config->urlGet . '/' . $id, ['max_retry_attempts' => 5, 'retry_on_status'=> [400]]);

            if ($response->getStatusCode() == 200) {
                $result = json_decode($response->getBody()->getContents(), true);
                $order = $this->orderService->FindByField('fiscalProtocolo', $id);
                if (!$order) {
                    throw new NotFoundException('Pedido não localizado!');
                }

                $order->chaveFiscal = $result['id'];
                $order->fiscalStatus = $result['status'];
                $order->save();
                if ($this->config->autoPrint) {
                    $this->client->get($this->config->urPrint . '/' . $order->chaveFiscal);
                }
                event(new DeliveryOrderCfeCreated($order));
                return $order;
            }
        } catch (ClientException $e) {
            throw new Exception($e->getMessage());
        }
    }

    private function makeItens($itens)
    {
        foreach ($itens as $it) {

            $item = [
                "service" => 0,
                "service_apportion" => 0,
                "cfop" => $it['cfop'],
                "code" => $it['codigo'],
                "item" => $it['nrItem'],
                "name" => $it['descricao'],
                "ncm" => $it['ncm'],
                "quantity" => $it['nrQuantidade'],
                "price" => $it['vrUnitario'],
                "unit" => $it['medida'],
                "tax" => [
                    "cofins" => [
                        "aliquot" => $it['aliquotaCofins'],
                        "cst" => $it['cstCofins']
                    ],
                    "pis" => [
                        "aliquot" => $it['aliquotaPis'],
                        "cst" => $it['cstPis']
                    ],
                    "ibpt" => (floatval($it['ibpt']) / 100) * floatval($it['vrUnitario']),
                    "icms" => [
                        "aliquot" => $it['aliquotaIcms'],
                        "cst" => $it['cstIcms']
                    ]
                ]
            ];

            array_push($this->venda['products'], $item);
        }
    }

    private function makePayment($payments)
    {
        foreach ($payments as $pay) {
            $formadePagamento = FormaPagamento::find($pay->idFormaPagamento);
            $payment = [
                "accrediting" => null,
                "code" => $formadePagamento['pagamentoSat'],
                "value" => $pay['vrFormaPagamento']
            ];
            array_push($this->venda['payments'], $payment);
        }
    }

}
