<?php

namespace App\Services\Nati;

use Exception;
use Carbon\Carbon;
use App\Entities\User;
use App\Services\Service;
use App\Entities\Nati\OrderTab;
use App\Entities\Nati\OrderItem;
use Illuminate\Support\Facades\DB;
use App\Entities\Nati\OrderDeletedTab;
use Illuminate\Database\Eloquent\Model;
use App\Entities\Nati\OrderDeletedItem;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Validation\UnauthorizedException;
use Illuminate\Database\Eloquent\ModelNotFoundException;

class OrderTabService extends Service
{
    /**
     * @return mixed
     */
    public function getAll($request)
    {
        $block = filter_var($request['block'], FILTER_VALIDATE_BOOLEAN);

        return OrderTab::withCount('items')
            ->where('flBloqueada', $block)
            ->get();
    }

    /**
     * @param string $tabId
     * @return Builder|Builder[]|Collection|Model|null
     */
    public function show(string $tabId)
    {
        return OrderTab::with('items')->find($tabId);
    }

    /**
     * @param string $tabId
     * @param array  $request
     * @return bool
     * @throws UnauthorizedException|
     * @throws Exception
     */
    public function delete(string $tabId, array $request)
    {
        try {
            $user = User::where(function ($query) use ($request) {
                $query->where('idOperador', $request['usuario'])
                    ->orWhere('stApelido', $request['usuario']);
            })
                ->where('stSenha', md5($request['senha']))
                ->firstOrFail();
        } catch (ModelNotFoundException $exception) {
            throw new ModelNotFoundException("Usuário ou senha incorreto");
        }

        if (!$user->hasAcesso(1)) {
            throw new UnauthorizedException("Você não tem autorização para realizar essa ação");
        }

        $tab = OrderTab::find($tabId);

        if ($tab->idMicroterminallock != '') {
            throw new UnauthorizedException("Comanda travada pelo terminal" . $tab->idMicroterminallock);
        }

        $items = OrderItem::where('idComanda', $tabId)->get();

        $id = $user->idOperador;

        $idAtendimento = OrderDeletedTab::max('idAtendimento') + 1;

        $exclude = [
            'idAtendimento' => $idAtendimento,
            'idComanda' => $tab->idComanda,
            'dtFirstDate' => $tab->dtFirstDate,
            'dtLastdate' => $tab->dtLastdate,
            'flReturn' => $tab->flReturn,
            'idOperadorCaixa' => 0,
            'idOperadorExcluiu' => $id,
            'dtAlteracao' => Carbon::now()->toDateTimeString(),
        ];

        $itemsExcluir = $items->map(function ($item) use ($idAtendimento, $id) {
            return [
                'idAtendimento' => $idAtendimento,
                'nrItem' => $item->nrItem,
                'idProduto' => $item->idProduto,
                'stOperador' => $item->stOperador,
                'nrQuantidade' => $item->nrQuantidade,
                'dtData' => $item->dtData,
                'idMicroterminal' => $item->idMicroterminal,
                'vrUnitario' => $item->vrUnitario,
                'vrTotal' => $item->vrTotal,
                'dtAlteracao' => Carbon::now()->toTimeString(),
                'idOperadorExcluiu' => $id,
            ];
        })->toArray();

        DB::transaction(function () use ($exclude, $tabId, $itemsExcluir) {

            OrderDeletedTab::insert($exclude);

            OrderDeletedItem::insert($itemsExcluir);

            OrderItem::where('idComanda', $tabId)->delete();

            OrderTab::find($tabId)->delete();
        });

        return true;
    }

    /**
     * @param string $tabId
     * @param array  $request
     * @return bool
     * @throws UnauthorizedException|
     */
    public function lock(string $tabId, array $request)
    {
        try {
            if(isset($request['usuario'])){
                $user = User::where(function ($query) use ($request) {
                    $query->where('idOperador', $request['usuario'])
                        ->orWhere('stApelido', $request['usuario']);
                })
                    ->where('stSenha', md5($request['senha']))
                    ->firstOrFail();
            }else{
                $user = auth()->user();
            }

        } catch (ModelNotFoundException $exception) {
            throw new ModelNotFoundException("Usuário ou senha incorreto");
        }

        $permission = json_decode($user->perfil->stIds);

        if (!$user->hasAcesso(31)) {
            throw new UnauthorizedException("Você não tem autorização para realizar essa ação");
        }

        $tab = OrderTab::find($tabId);

        if($tab){
            if ($tab->idMicroterminallock != '') {
                throw new UnauthorizedException("Comanda travada pelo terminal" . $tab->idMicroterminallock);
            }

            OrderTab::where('idComanda', $tabId)->update([
                'flBloqueada' => true,
                'dtAlteracao' => Carbon::now()->toDateTimeString()
            ]);
        }else{
            $order = new OrderTab();
            $order->idComanda =$tabId;
            $order->flBloqueada = true;
            $order->save();
        }



        return true;
    }

    /**
     * @param string $tabId
     * @param array  $request
     * @return bool
     * @throws UnauthorizedException|
     */
    public function unlock(string $tabId, array $request)
    {
        try {
            $user = User::where(function ($query) use ($request) {
                if(is_numeric($request['usuario'])) {
                    $query->where('idOperador', $request['usuario']);
                } else {
                    $query->where('stApelido', $request['usuario']);
                }
            })
                ->where('stSenha', md5($request['senha']))
                ->firstOrFail();
        } catch (ModelNotFoundException $exception) {
            throw new ModelNotFoundException("Usuário ou senha incorreto");
        }

        $permission = json_decode($user->perfil->stIds);

        if (!$user->hasAcesso(31)) {
            throw new UnauthorizedException("Você não tem autorização para realizar essa ação");
        }

        $tab = OrderTab::find($tabId);

        if ($tab->idMicroterminallock != '') {
            throw new UnauthorizedException("Comanda travada pelo terminal" . $tab->idMicroterminallock);
        }

        OrderTab::where('idComanda', $tabId)->update([
            'flBloqueada' => false,
            'dtAlteracao' => Carbon::now()->toDateTimeString()
        ]);

        return true;
    }

    public function getItems(string $tabId)
    {
        $res = OrderTab::find($tabId)->items()->with('item')->get();

        /**
        * $item = mt_Itens
        * $item->item = prd_Produtos
        */
        return $res->map(function ($item) {
            $medida = $item->item->medida()->first();
            return [
                'code' => $item->item->code,
                'description' => $item->item->stProduto,
                'medida' => $medida->stMedida,
                'quantity' => $medida->flFracionado ? $item->nrQuantidade / 1000 : $item->nrQuantidade,
                'value' => $item->vrUnitario,
                'waiter' => $item->stOperador,
                'date' => $item->dtData,
                'total_value' => $item->vrTotal
            ];
        });
    }
}
