<?php

namespace App\Services\Nati;

use App\Entities\Financeiro\ContaAPagar;
use App\Entities\Nati\StockBalance;
use App\Entities\Nati\StockItemsMovimentation;
use App\Entities\Nati\StockMovimentation;
use App\Entities\Supplier;
use App\Repositories\Nati\ProductRepository;
use App\Repositories\Nati\StockBalanceRepository;
use App\Services\Service;
use App\Validators\Nati\StockBalanceValidator;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\UnauthorizedException;

class StockBalanceService extends Service
{
    protected $repository;
    protected $validator;
    protected $productRepository;

    public function __construct(StockBalanceRepository $repository, StockBalanceValidator $validator, ProductRepository $productRepository)
    {
        $this->repository = $repository;
        $this->validator = $validator;
        $this->productRepository = $productRepository;
    }

    public function get()
    {
        return $this->repository->with('produto')->all(['idSaldo', 'idProduto', 'final_balance']);
    }

    public function insert(array $invoice)
    {
        $user = Auth::user();

        if (!$user->hasAcesso(48)) {
            throw new UnauthorizedException("Você não possui permissão para realizar lançamentos");
        }

        try {
            DB::beginTransaction();

            $supplier = Supplier::find($invoice['origem']);
            $supplier->dtUltimaCompra = Carbon::now()->toDateTimeString();
            $supplier->save();

            $idMovimentacao = StockMovimentation::max('idMovimentacao') + 1;

            StockMovimentation::insert([
                'idMovimentacao' => $idMovimentacao,
                'idOrigem' => $invoice['origem'],
                'idDestino' => 1,
                'dtMovimentacao' => Carbon::now()->toDateTimeString(),
                'stOperador' => $user->stApelido,
                'stNotaFiscal' => $invoice['fiscalNumber'],
                'flEntrada' => true,
                'flTipoOrigem' => true,
                'flTipoDestino' => true,
                'flAtivo' => true,
                'stObservacao' => $invoice['obs'],
                'dtData' => Carbon::createFromFormat("d/m/Y", $invoice['date'])->toDateTimeString(),
                'flTipoEstoque' => true
            ]);

            $idItem = StockItemsMovimentation::max('idItem') + 1;

            $itemsMovimentation = [];

            foreach ($invoice['products'] as $product) {
                $this->productRepository->update([
                    'vrCusto' => $product['custo'],
                    'dtUltimaCompra' => Carbon::now()->toDateTimeString()
                ], $product['id']);

                StockItemsMovimentation::create([
                    'idItem' => $idItem++,
                    'idMovimentacao' => $idMovimentacao,
                    'idProduto' => $product['id'],
                    'nrQuantidade' => $product['qtd'],
                    'vrValor' => $product['qtd'] * $product['custo'],
                    'stMedidaConversao' => ' ',
                    'nrQuantidadeConversao' => ' '
                ]);
//array_push($itemsMovimentation, [
//                    'idItem' => $idItem++,
//                    'idMovimentacao' => $idMovimentacao,
//                    'idProduto' => $product['id'],
//                    'nrQuantidade' => $product['medida']['flFracionado'] ? $product['qtd'] * 1000 : $product['qtd'],
//                    'vrValor' => $product['qtd'] * $product['custo'],
//                    'stMedidaConversao' => 'N',
//                    'nrQuantidadeConversao' => '0'
//                ]);

                $qtd =  $product['qtd'];

                StockBalance::where('idProduto', $product['id'])->update([
                    'nrQuantidade' => DB::raw("nrQuantidade + $qtd"),
                    'dtModificacao' => Carbon::now()->toDateTimeString(),
                    'dtConferencia' => Carbon::now()->toDateTimeString(),
                ]);
            }

//            StockItemsMovimentation::insert($itemsDocumentation);

            if ($invoice['finance']) {
                $idConta = ContaAPagar::where('idFornecedor', $invoice['origem'])->max('idContasPagar') + 1;
                $contas = [];

                $col = collect($itemsMovimentation);
                $total = $col->reduce(function ($val, $item) {
                    return $val + $item['vrValor'];
                }, 0);

                for ($i = 0; $i < $invoice['financeForm']['parts']; $i++) {
                    array_push($contas, [
                        'idContasPagar' => $idConta++,
                        'idTipoConta' => $invoice['financeForm']['subgroup'],
                        'idFornecedor' => $invoice['origem'],
                        'idCentroCusto' => $invoice['financeForm']['cost_center'],
                        'dtLancamento' => Carbon::createFromFormat('d/m/Y', $invoice['date'])->toDateTimeString(),
                        'dtVencimento' => Carbon::createFromFormat('d/m/Y', $invoice['financeForm']['due_date'])->toDateTimeString(),
                        'nrParcela' => $i + 1,
                        'nrTotalParcelas' => $invoice['financeForm']['parts'],
                        'vrJuros' => 0,
                        'stOperador' => $user->stApelido,
                        'stObservacao' => $invoice['obs'],
                        'stDocumento' => $invoice['fiscalNumber'],
                        'vrDocumento' => $total,
                        'idAbertura' => 0,
                        'idTipoPagamento' => $invoice['financeForm']['payment'],
                        'idMovimentacao' => $idMovimentacao,
                        'idConta' => $invoice['financeForm']['account'],
                        'vrEncargos' => $invoice['financeForm']['taxes'],
                        'vrOutros' => $invoice['financeForm']['otherExpenses'],
                        'vrDescontos' => $invoice['financeForm']['discount']
                    ]);
                }
                ContaAPagar::insert($contas);
            }
            DB::commit();
        } catch (\Exception $exception) {
            DB::rollBack();
            throw $exception;
        }
    }

    public function createStockBalance(int $idProduto)
    {

        $existStockBalance = StockBalance::select(['idSaldo'])
                                ->where('idProduto', '=', $idProduto)
                                ->get();
        if(!empty($existStockBalance)) {
            $stock = StockBalance::create([
                'idSaldo' => StockBalance::max('idSaldo') + 1,
                'idEstoque' => '1',
                'idProduto' => $idProduto,
                'nrQuantidade' => 0,
                'nrReservas' => 0,
                'dtModificacao' => Carbon::now()->toTimeString(),
                'dtConferencia' => Carbon::now()->toTimeString(),
                'nrMinimo' => 0,
                'dtAlteracao' => Carbon::now()->toTimeString(),
            ]);
            $stock->save();
        }
    }

    public function deleteStockBalance(int $idProduto)
    {
        $existStockBalance = StockBalance::select(['idSaldo'])
                                ->where('idProduto', '=', $idProduto)
                                ->get();
        if(!empty($existStockBalance)) {
            $stock = StockBalance::where('idProduto', '=', $idProduto)
                                    ->delete();

        }
    }
}
