<?php

namespace App\Services\Delivery;

use Carbon\Carbon;
use App\Events\OrderCreated;
use App\Entities\Nati\Product;
use App\Entities\Delivery\Item;
use App\Entities\Delivery\Order;
use Illuminate\Support\Facades\DB;
use App\Entities\Delivery\Payment;
use App\Entities\Nati\FormaPagamento;
use App\Entities\Delivery\MobyoConfig;
use App\Entities\Mobyo\MobyoPayment;
use App\Repositories\DlvOrderRepository;
use App\Validators\Delivery\OrderValidator;
use App\Validators\Delivery\IfoodOrderValidator;
use mysql_xdevapi\Exception;
use Prettus\Validator\Contracts\ValidatorInterface;

class MobyoIntegrationService
{
    protected $repository;
    protected $cepService;
    protected $customerService;
    protected $printService;
    protected $orderValidator;
    protected $ifoodOrderValidator;
    protected $config;
    protected $engine;
    protected $integration;
    protected $mobyoRequest;
    protected $orderDelivery;
    protected $orderMobyo;
    public function __construct(
        DlvOrderRepository $repository,
        CustomerService $customerService,
        CepService $cepService,
        PrintService $printService,
        OrderValidator $orderValidator,
        IfoodOrderValidator $ifoodOrderValidator
        )
    {
        $this->repository = $repository;
        $this->cepService = $cepService;
        $this->customerService = $customerService;
        $this->printService = $printService;
        $this->orderValidator = $orderValidator;
        $this->ifoodOrderValidator = $ifoodOrderValidator;
    }


    private function trataString($string)
    {
        $comAcentos = array('à', 'á', 'â', 'ã', 'ä', 'å', 'ç', 'è', 'é', 'ê', 'ë', 'ì', 'í', 'î', 'ï', 'ñ', 'ò', 'ó', 'ô', 'õ', 'ö', 'ù', 'ü', 'ú', 'ÿ', 'À', 'Á', 'Â', 'Ã', 'Ä', 'Å', 'Ç', 'È', 'É', 'Ê', 'Ë', 'Ì', 'Í', 'Î', 'Ï', 'Ñ', 'Ò', 'Ó', 'Ô', 'Õ', 'Ö', 'O', 'Ù', 'Ü', 'Ú');
        $semAcentos = array('a', 'a', 'a', 'a', 'a', 'a', 'c', 'e', 'e', 'e', 'e', 'i', 'i', 'i', 'i', 'n', 'o', 'o', 'o', 'o', 'o', 'u', 'u', 'u', 'y', 'A', 'A', 'A', 'A', 'A', 'A', 'C', 'E', 'E', 'E', 'E', 'I', 'I', 'I', 'I', 'N', 'O', 'O', 'O', 'O', 'O', '0', 'U', 'U', 'U');

        return str_replace($comAcentos, $semAcentos, $string);
    }

    public function handleRequest($request)
    {
        if ($request['engineType'] === 'ANOTEAKI') {
            $deliveryAddress = [
                'formattedAddress' => "0",
                'country' => "0",
                'state' => "0",
                'city' => "0",
                'coordinates' => [
                    'latitude' => 0,
                    "longitude" => 0
                ],
                "neighborhood" => "0",
                "streetName" => "0",
                "streetNumber" => "0",
                "postalCode" => "0",
                "reference" => "0"
            ];
            $request['deliveryAddress'] = $deliveryAddress;

            foreach ($request['items'] as $key => $item) {
                if (!empty($item['subItems'])) {
                    foreach ($item['subItems'] as $keySub => $subItem) {
                        if (isset($subItem['price'])) {
                            $request['items'][$key]['subItems'][$keySub]['totalPrice'] = $subItem['price'];
                        }
                    }
                }
            }
            return $request;
        }


        if ($request['engineType'] === 'MOBYO' || $request['engineType'] === 'IFOOD' && $request['orderType'] === 'TAKEOUT' || $request['engineType'] === 'EPADOCA' && $request['orderType'] === 'TAKEOUT') {
            $deliveryAddress = [
                'formattedAddress' => "0",
                'country' => "0",
                'state' => "0",
                'city' => "0",
                'coordinates' => [
                    'latitude' => 0,
                    "longitude" => 0
                ],
                "neighborhood" => "0",
                "streetName" => "0",
                "streetNumber" => "0",
                "postalCode" => "0",
                "reference" => "0"
            ];
            $request['delivery']['deliveryAddress'] = $deliveryAddress;
            return $request;
        }

        if ($request['engineType'] === 'ZEDELIVERY' && !$request['delivery']['deliveryAddress']) {
            $deliveryAddress = [
                'formattedAddress' => "0",
                'country' => "0",
                'state' => "0",
                'city' => "0",
                'coordinates' => [
                    'latitude' => 0,
                    "longitude" => 0
                ],
                "neighborhood" => "0",
                "streetName" => "0",
                "streetNumber" => "0",
                "postalCode" => "0",
                "reference" => "0"
            ];
            $request['delivery']['deliveryAddress'] = $deliveryAddress;

            foreach ($request['items'] as $key => $item) {
                if (!empty($item['subItems'])) {
                    foreach ($item['subItems'] as $keySub => $subItem) {
                        if (isset($subItem['price'])) {
                            $request['items'][$key]['subItems'][$keySub]['totalPrice'] = $subItem['price'];
                        }
                    }
                }
            }
            return $request;
        }

        if ($request['engineType'] === 'ZEDELIVERY' && $request['delivery']['deliveryAddress']['postalCode'] == '') {
            $request['delivery']['deliveryAddress']['postalCode'] = "0";
            return $request;
        }

        return $request;
    }

    public function isIfoodmercado()
    {
        if($this->mobyoRequest['engineName'] === 'IFOOD MERCADO') {
            return true;
        }
        return false;
    }

    public function getIntegrationSettings()
    {
        try {

            if($this->isIfoodmercado()) {
                $integration = MobyoConfig::where('engine', $this->mobyoRequest['engineName'])
                                            ->with('payments')
                                            ->first();
            } else {
                $integration = MobyoConfig::where('engine', $this->mobyoRequest['engineType'])
                                            ->with('payments')
                                            ->first();
            }
            if (!$integration['flAtivo']) {
             throw new \Exception('Nati: Engine não habilitada');
            } elseif ($integration['idCliente'] === null && $integration['flClienteFixo']) {
             throw new \Exception('Nati: Id do Cliente não pode ser nulo');
            } elseif ($integration['codProdutoGenerico'] === null) {
             throw new \Exception('Nati: Codigo do produto generico não pode ser nulo');
            }
            return $integration;
        } catch (\Exception $e) {
            throw new \Exception($e);
        }

    }

    public function save($req)
    {
        if(!isset($req['delivery']['deliveryAddress']['state'])) {
            $req['delivery']['deliveryAddress']['state'] = 'SP';
        }
        $this->mobyoRequest = $this->handleRequest($req);
        $this->settingsMobyo = $this->getIntegrationSettings();
        $this->ifoodOrderValidator->with($this->mobyoRequest)->passesOrFail(ValidatorInterface::RULE_CREATE);
        $exist = $this->repository->findByField('reference', $this->mobyoRequest['reference'])->first();
        if ($exist) {
            return $exist;
        }
        try {
            $this->orderMobyo = new \App\Entities\Mobyo\Order($this->mobyoRequest);

            DB::beginTransaction();

            if($this->settingsMobyo['engine'] === 'ZEDELIVERY') {
                $this->createDLVOrderWithoutCPF();
            } else {
                $this->createDLVOrder();
            }

            $this->insertItems();
            if ($this->orderDelivery->items()->count() == 0) throw new \Exception("A ordem não tem item");
            $this->insertDeliveryTax();
            $this->insertServiceTax();
            $this->insertPayments();

            DB::commit();

            event(new OrderCreated($this->orderDelivery));

            return $this->orderDelivery;
        } catch (\Exception $e) {
            DB::rollBack();
            throw new \Exception($e);
        }
    }

    private function filterFracionado(array $data)
    {
        $res = array_filter($data, function ($it) {
            if (isset($it['externalCode'])) {
                $externalCode = isset($it['externalCode']) && $it['externalCode'] != '0' ? $it['externalCode'] : $this->settingsMobyo['codProdutoGenerico'];
                $produto = $this->getIdProdutoByCode($externalCode);
                return $produto['idTipoProduto'] == '3';
            }
        });

        $res = $this->formatArray($res);

        return $res;
    }

    private function filterNormal(array $data)
    {
        $res = array_filter($data, function ($it) {
            if (isset($it['externalCode'])) {
                $externalCode = isset($it['externalCode']) && $it['externalCode'] != '0' ? $it['externalCode'] : $this->settingsMobyo['codProdutoGenerico'];
                $produto = $this->getIdProdutoByCode($externalCode);
                return $produto['idTipoProduto'] != '3';
            }
        });

        if(!$res) {
            foreach ($data as $item) {
                if (!isset($item['externalCode'])) {
                    $item['externalCode'] = $this->settingsMobyo['codProdutoGenerico'];
                    $item['isPizza'] = 1;
                    $res[] = $item;
                }
            }
        }

        return $res;
    }

    private function formatArray($array) {
        $fracionados = [];
        foreach($array as $value) {
            array_push($fracionados, $value);
        }
        return $fracionados;
    }

    private function makeItem($data, $idAtendimento, $count, $produto = null, $quantidade = 1, $codProdutoGenerico = '-100')
    {
        try {
            if ($produto) {
                $qtd = floatval($data['quantity']) * $quantidade;
                $observacoes = $produto['codigo'] == $codProdutoGenerico ? $this->trataString($data['name']) : '';
                if (isset($data['observations'])) {
                    $observacoes .= ' ' . $this->trataString($data['observations']);
                }
                $unitario = floatval($data['unitPrice']) > 0 ? floatval($data['unitPrice']) : 0;
                $total = $data['price'];

                $item = new Item([
                    'idProduto' => $produto['idProduto'],
                    'stOperador' => 'INTEGRACAO',
                    'nrQuantidade' => $qtd,
                    'dtData' => Carbon::now(),
                    'vrUnitario' => $unitario,
                    'vrTotal' => number_format($total, 2, ".", ""),
                    'idAtendimento' => $idAtendimento,
                    'nrItem' => $count,
                    'stObservacoes' => '',
                    'stValores' => "",
                    'stCodigos' => "",
                    'stQuantidades' => "",
                    'flPrint' => true,
                    'idComanda' => "",
                    'dtAlteracao' => Carbon::now(),
                    'flAtivo' => true,
                    'stOBS' => $observacoes,
                    'codigo' => $produto['codigo'],
                    'descricao' => $produto['descricao'],
                    'cfop' => $produto['cfop'],
                    'ncm' => $produto['ncm'],
                    'cstIcms' => $produto['cstIcms'],
                    'aliquotaIcms' => number_format($produto['aliquotaIcms'], 2),
                    'cstPis' => $produto['cstPis'],
                    'aliquotaPis' => number_format($produto['aliquotaPis'], 2),
                    'cstCofins' => $produto['cstCofins'],
                    'aliquotaCofins' => number_format($produto['aliquotaCofins'], 2),
                    'ibpt' => number_format($produto['ibpt'] / 100 * $total, 2),
                    'medida' => $produto['medida'],
                ]);
                return $item;
            }
        } catch (\Exception $e) {
            throw new \Exception('NATI: Item nao encontrado');
        }
    }

    private function getCustomer($data)
    {
        if ($this->settingsMobyo['flClienteFixo']) {
            try {
                return $this->settingsMobyo['idCliente'];
            } catch (\Exception $e) {
                throw  new \Exception("NATI: Cliente não configurado");
            }
        } else {
            $customer = $this->customerService->getByName($data['customer']['name']);
            if (!$customer) {
                $customer = $this->customerService->create([
                    'stCliente' => $data['name'],
                    'flPessoaJuridica' => false,
                    'nrNumero' => $data['deliveryAddress']['streetNumber'],
                    'idCEP' => $this->GetIdCep($data['deliveryAddress']),
                    'flBloqueado' => false,
                    'flAtivo' => true,
                    'dtCadastro' => Carbon::now(),
                    'stOperador' => 'IFOOD',
                    'dtUltimaAlteracao' => Carbon::now(),
                    'stOperadorUltimaAlteracao' => 'MOBYO',
                ]);
            }
            return $customer->idCliente;
        }
    }

    private function getIdProdutoByCode($codigo)
    {
        $codigo = ltrim($codigo, '0');
        try {
            $produto = Product::select([
                'prd_Produtos.idProduto',
                'prd_Codigos.idCodigo as codigo',
                'prd_produtos.stProdutoAbreviado as descricao',
                'prd_produtos.nrNcm as ncm',
                'prd_Aliquotas.cfopSaida as cfop',
                'prd_Aliquotas.CstIcms as cstIcms',
                'prd_Aliquotas.AliquotaIcms as aliquotaIcms',
                'prd_Aliquotas.CstPis as cstPis',
                'prd_Aliquotas.AliquotaPis as aliquotaPis',
                'prd_Aliquotas.CstCofins as cstCofins',
                'prd_Aliquotas.AliquotaCofins as aliquotaCofins',
                'prd_Medidas.stMedida as medida',
                DB::raw("COALESCE(tbt_CargaTributariaMedia.vrAliqNac, 4.20) AS ibpt"),
                'prd_Produtos.idTipoProduto',
            ])
                ->join('prd_codigos', 'prd_produtos.idProduto', '=', 'prd_Codigos.idProduto')
                ->join('prd_Aliquotas', 'prd_produtos.idAliquota', '=', 'prd_Aliquotas.idAliquota')
                ->join('prd_Medidas', 'prd_produtos.idMedida', '=', 'prd_Medidas.idMedida')
                ->leftJoin('tbt_CargaTributariaMedia', 'prd_produtos.nrNcm', '=', 'tbt_CargaTributariaMedia.stCodigo')
                ->where(function ($query) use ($codigo) {
                    $query->where('prd_Codigos.idCodigo', '=', "$codigo");
                    $query->where('prd_produtos.flAtivo', '=', true);
                })->first();

            return $produto;
        } catch (\Exception $e) {
            throw  new \Exception("NATI: Codigo do produto não encontrado");
        }
    }

    private function getNextId()
    {
        return Order::max('idAtendimento') + 1;
    }

    private function GetIdCep($data)
    {
        $cep = $this->cepService->getByCep($data['postalCode']);
        if ($cep) {
            return $cep;
        } else {
            $res = $this->cepService->create([
                'stCEP' => $data['postalCode'],
                'stEndereco' => $data['streetName'],
                'stBairro' => $data['neighborhood'],
                'stCidade' => $data['city'],
                'stEstado' => $data['state'],
            ]);
            return $res->idCEP;
        }
    }

    public function dispatchOrReceive($request)
    {
        DB::transaction(function () use ($request) {
            $id = "";

            for ($x = 0; $x < strlen($request['stCupomDelivery']); $x++) {
                if (ctype_digit($request['stCupomDelivery'][$x]))
                    $id .= $request['stCupomDelivery'][$x];
            }

            $order = Order::findOrFail($id);

            if (!$order->flAtivo)
                throw new \Exception("O pedido informado está excluido.");
            elseif ($order->flOrcamento)
                throw new \Exception("Você não pode enviar um orçamento para entrega.");
            else {
                switch ($order->stStatus) {
                    case 'F':
                        throw new \Exception("O pedido informado está em formulação.");
                    case 'E':
                        throw new \Exception("O pedido informado já foi entregue.");
                    case 'P':
                        if ($request['statusDr'] != 2) {
                            $order->stStatus = 'S';
                            $order->dtSaidaMotoboy = Carbon::now()->toTimeString();
                            $order->idMotoboy = $request['idMotoboy'];
                            $order->save();
                            return [
                                'message' => "O pedido foi enviado com sucesso."
                            ];
                        } else {
                            throw new \Exception("O pedido informado ainda não foi despachado.");
                        }
                    case 'S':
                        if ($request['statusDr'] != 1) {
                            $order->stStatus = 'E';
                            $order->dtSaidaMotoboy = Carbon::now()->toTimeString();
                            $order->save();
                            return [
                                'message' => "O pedido foi finalizado com sucesso."
                            ];
                        } else {
                            throw new \Exception("O pedido informado já foi despachado.");
                        }
                }
            }
        });
    }

    public function FindByField($field, $value)
    {
        return $this->repository->findByField($field, $value)->first();
    }

    private function insertPaymentWithChangefor($order, $pay, $idFormaPagamento)
    {
        $payment = new Payment();
        $payment->setId($order->idAtendimento);
        $payment->idFormaPagamento = $idFormaPagamento;
        $payment->vrFormaPagamento = $pay['changeFor'];
        $payment->dtData = Carbon::now();
        $payment->dtAlteracao = Carbon::now();
        $order->payments()->save($payment);

        $changeFor = new Payment();
        $changeFor->flTroco = true;
        $changeFor->setId($order->idAtendimento);
        $changeFor->idFormaPagamento = $idFormaPagamento;
        $changeFor->vrFormaPagamento = $pay['changeFor'] - $pay['value'];
        $changeFor->dtData = Carbon::now();
        $changeFor->dtAlteracao = Carbon::now();

        $order->payments()->save($changeFor);
    }

    private function getObs()
    {
        $dt = $this->orderMobyo->getCreatedAt();
        if ($this->settingsMobyo['flSalvarClienteObservacao']) {
            $obs = 'CLIENTE: ' . $this->orderMobyo->getCustomerName() . chr(10)
                . 'TELEFONE: ' . $this->orderMobyo->getPhone() . chr(10)
                . 'PEDIDO: #' . $this->orderMobyo->getShortReference() . chr(10)
                . 'FEITO AS ' . $dt->hour . ':' . $dt->minute . chr(10)
                . 'LOCALIZADOR DO PEDIDO ' . $this->orderMobyo->getLocalizador() . chr(10)
                . 'ENTREGA PREVISTA: ' . ($this->orderMobyo->getDeliveryDateTime() ?? "") . chr(10)
                . $this->orderMobyo->getOrdersCount() . ' PEDIDOS NA SUA LOJA' . chr(10)
                . 'ENTREGAR EM: ' . $this->orderMobyo->getFormatedAddress() . $this->orderMobyo->getAddress('neighborhood') . chr(10)
                . $this->orderMobyo->getAddress('city') . ' - CEP: ' . $this->orderMobyo->getAddress('postalCode') . chr(10)
                . $this->orderMobyo->getAddressReference() . chr(10)
                . $this->orderMobyo->getComplemento() . chr(10);
        }

        return $obs;
    }

    private function createDLVOrder()
    {
        $this->orderDelivery = Order::create([
                                'idAtendimento' => $this->getNextId(),
                                'idCliente' => $this->getCustomer($this->mobyoRequest),
                                'idOperador' => $this->settingsMobyo['idUser'],
                                'dtAbertura' => Carbon::now(),
                                'dtConclusao' => Carbon::now(),
                                'stStatus' => $this->settingsMobyo['flDespacharPedidoAutomaticamente'] && $this->mobyoRequest['orderTiming'] != 'SCHEDULED' ? 'S' : 'P',
                                'stOBS' => $this->getObs($this->orderMobyo),
                                'reference' => $this->orderMobyo->getReference(),
                                'flPago' => false,
                                'flClienteRetira' => $this->orderMobyo->getClienteRetira(),
                                'flAgendarPedido' => ($this->orderMobyo->getScheduled() ?? false),
                                'dtAgendarPedido' => $this->orderMobyo->getDeliveryDateTime(),
                                'dtAgendarPedidoProducao' => ($this->orderMobyo->getPreparationStartDateTime() ?? ""),
                                'desconto' => $this->orderMobyo->getDesconto(),
                                'cliente' => $this->orderMobyo->getCustomer(),
                                'stCpf' => $this->orderMobyo->getDocumentNumber(),
                                'engine' => $this->orderMobyo->getEngineType()
                            ]);
    }

    private function createDLVOrderWithoutCPF()
    {
        $this->orderDelivery = Order::create([
                                'idAtendimento' => $this->getNextId(),
                                'idCliente' => $this->getCustomer($this->mobyoRequest),
                                'idOperador' => 999,
                                'dtAbertura' => Carbon::now(),
                                'dtConclusao' => Carbon::now(),
                                'stStatus' => $this->settingsMobyo['flDespacharPedidoAutomaticamente'] && $this->mobyoRequest['orderTiming'] != 'SCHEDULED' ? 'S' : 'P',
                                'stOBS' => $this->getObs(),
                                'reference' => $this->orderMobyo->getReference(),
                                'flPago' => false,
                                'flClienteRetira' => $this->orderMobyo->getClienteRetira(),
                                'flAgendarPedido' => ($this->orderMobyo->getScheduled() ?? false),
                                'dtAgendarPedido' => $this->orderMobyo->getDeliveryDateTime(),
                                'dtAgendarPedidoProducao' => ($this->orderMobyo->getPreparationStartDateTime() ?? ""),
                                'desconto' => $this->orderMobyo->getDesconto(),
                                'cliente' => $this->orderMobyo->getCustomer(),
                                'engine' => $this->orderMobyo->getEngineType()
                            ]);
    }

    private function insertItems()
    {
        $count = 1;

        foreach ($this->mobyoRequest['items'] as $it) {
            $externalCode = isset($it['externalCode']) ? $it['externalCode'] : $this->settingsMobyo['codProdutoGenerico'];
            $produto = $this->getIdProdutoByCode($externalCode);
            if (!$produto) {
                $produto = $this->getIdProdutoByCode($this->settingsMobyo['codProdutoGenerico']);
            }
            if (!$produto) {
                throw new \Exception('Não foi possivel localizar o produto');
            }

            if ($produto['idTipoProduto'] == '3' && isset($it['options'])) {
                $frac = $this->filterFracionado($it['options']);
                $normal = $this->filterNormal($it['options']);
                if ($frac && $this->settingsMobyo['engine'] == 'EPADOCA') {
                    $observacoes = $it['skill'] == 'observation' ? $this->trataString($it['name']) : '';
                    $produto = $this->getIdProdutoByCode($frac[0]['externalCode']);
                    $item = $this->makeItem($frac[0], $this->orderDelivery->idAtendimento, $count, $produto, $it['quantity']);
                    $item->stOBS = $observacoes;
                    $total = 0;
                    if (count($frac) == 1) {
                        $produto = $this->getIdProdutoByCode($frac[0]['externalCode']);
                        $item->idProduto = $produto['idProduto'];
                        $item->nrQuantidade = $frac[0]['quantity'];
                    } else {
                        foreach ($frac as $i) {
                            $produto = $this->getIdProdutoByCode($i['externalCode']);
                            if ($produto) {
                                if(!isset($i['totalPrice'])) $i['totalPrice'] = $i['quantity'] * $i['price'];
                                $total += $i['totalPrice'];
                                $item->stCodigos .= (strlen($item->stCodigos) >= 1 ? "@" : "") . $produto['idProduto'];
                                $item->stValores .= (strlen($item->stValores) >= 1 ? "@" : "") . $i['totalPrice'];
                                $item->stQuantidades .= (strlen($item->stQuantidades) >= 1 ? "@" : "") . $i['quantity'] . '/' . count($frac);
                            }
                        }
                        $item->vrUnitario = $total;
                        $item->vrTotal = $total * $it['quantity'];
                    }

                    $this->orderDelivery->items()->save($item);
                    $count++;
                }
                if ($frac && $this->settingsMobyo['engine'] != 'EPADOCA') {
                    $observacoes = isset($it['observations']) ? $this->trataString($it['observations']) : '';
                    $produto = $this->getIdProdutoByCode($frac[0]['externalCode']);
                    $item = $this->makeItem($frac[0], $this->orderDelivery->idAtendimento, $count, $produto, $it['quantity']);
                    $item->stOBS = $observacoes;
                    $total = 0;
                    if (count($frac) == 1) {
                        $produto = $this->getIdProdutoByCode($frac[0]['externalCode']);
                        $item->idProduto = $produto['idProduto'];
                        $item->nrQuantidade = $frac[0]['quantity'];
                    } else {
                        foreach ($frac as $i) {
                            $produto = $this->getIdProdutoByCode($i['externalCode']);
                            if ($produto) {
                                if(!isset($i['totalPrice'])) $i['totalPrice'] = $i['quantity'] * $i['price'];
                                $total += $i['totalPrice'];
                                $item->stCodigos .= (strlen($item->stCodigos) >= 1 ? "@" : "") . $produto['idProduto'];
                                $item->stValores .= (strlen($item->stValores) >= 1 ? "@" : "") . $i['totalPrice'];
                                $item->stQuantidades .= (strlen($item->stQuantidades) >= 1 ? "@" : "") . $i['quantity'] . '/' . count($frac);
                            }
                        }
                        $item->vrUnitario = $total;
                        $item->vrTotal = $total * $it['quantity'];
                    }

                    $this->orderDelivery->items()->save($item);
                    $count++;
                }
                if ($normal) {
                    foreach ($normal as $i) {
                        if (isset($i['externalCode'])) {
                            $produto = $this->getIdProdutoByCode($i['externalCode']);
                            if (!$produto) {
                                $produto = $this->getIdProdutoByCode($this->settingsMobyo['codProdutoGenerico']);
                            }
                            if ($produto) {

                                $item = $this->makeItem($i, $this->orderDelivery->idAtendimento, $count, $produto, 1);
                                if(isset($i['isPizza'])) $item->stOBS = $i['name'];
                                $this->orderDelivery->items()->save($item);
                                $count++;
                            }
                        }
                    }
                }
            } else {
                $item = $this->makeItem($it, $this->orderDelivery->idAtendimento, $count, $produto, 1);
                if ($item) {
                    if (isset($it['options'])) {
                        foreach ($it['options'] as $i) {
                            if(!isset($i['totalPrice'])) $i['totalPrice'] = $i['quantity'] * $i['price'];
                            if ($i['totalPrice'] == 0) {
                                if ($i['quantity'] > 1) {
                                    $item->stOBS .= ' ' . $i['quantity'] . ' - ';
                                }
                                $item->stOBS .= ' | ' . $i['name'];
                            }
                        }
                    }

                    $this->orderDelivery->items()->save($item);
                    $count++;
                }
                if (isset($it['options'])) {
                    foreach ($it['options'] as $i) {
                        if(!isset($i['totalPrice'])) $i['totalPrice'] = $i['quantity'] * $i['price'];
                        if ($i['totalPrice'] > 0) {
                            $externalCode = isset($i['externalCode'])
                                ? $i['externalCode']
                                : $this->settingsMobyo['codProdutoGenerico'];
                            $produto = $this->getIdProdutoByCode($externalCode);
                            if ($produto) {
                                $item = $this->makeItem($i, $this->orderDelivery->idAtendimento, $count, $produto, $it['quantity']);
                                if ($item) {
                                    $this->orderDelivery->items()->save($item);
                                    $count++;
                                }
                            } else {
                                $produto = $this->getIdProdutoByCode($this->settingsMobyo['codProdutoGenerico']);
                                $item = $this->makeItem($i, $this->orderDelivery->idAtendimento, $count, $produto, $it['quantity']);
                                if ($item) {
                                    $this->orderDelivery->items()->save($item);
                                    $count++;
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    private function insertDeliveryTax()
    {
        if ($this->settingsMobyo['flSalvarTaxaEntrega']) {
            if (isset($this->mobyoRequest['total']['deliveryFee']) && $this->mobyoRequest['total']['deliveryFee'] > 0) {
                $produto = $this->getIdProdutoByCode('-100');
                if ($produto) {
                    $item = new Item([
                        'idAtendimento' => $this->orderDelivery->idAtendimento,
                        'nrItem' => $this->orderDelivery->items()->count() + 1,
                        'idProduto' => $produto['idProduto'],
                        'stOperador' => 'INTEGRACAO',
                        'nrQuantidade' => 1,
                        'dtData' => Carbon::now(),
                        'vrUnitario' => $this->orderMobyo->getDeliveryFee(),
                        'vrTotal' => $this->orderMobyo->getDeliveryFee(),
                        'flPrint' => true,
                        'idComanda' => "",
                        'dtAlteracao' => Carbon::now(),
                        'flAtivo' => true,
                        'codigo' => $produto['codigo'],
                        'descricao' => $produto['descricao'],
                        'cfop' => $produto['cfop'],
                        'ncm' => $produto['ncm'],
                        'cstIcms' => $produto['cstIcms'],
                        'aliquotaIcms' => number_format($produto['aliquotaIcms'], 2),
                        'cstPis' => $produto['cstPis'],
                        'aliquotaPis' => number_format($produto['aliquotaPis'], 2),
                        'cstCofins' => $produto['cstCofins'],
                        'aliquotaCofins' => number_format($produto['aliquotaCofins'], 2),
                        'ibpt' => number_format($produto['ibpt'] / 100 * $this->orderMobyo->getDeliveryFee(), 2),
                        'medida' => $produto['medida'],
                    ]);
                } else {
                    throw new \Exception("Erro ao inserir taxa de entrega: codigo '-100' não cadastrado");
                }
                $this->orderDelivery->items()->save($item);
            }
        }
    }

    private function insertServiceTax()
    {
        if (!$this->settingsMobyo['flSalvarTaxaServico']) {
            return;
        }
        if (isset($this->mobyoRequest['total']['additionalFees']) && $this->mobyoRequest['total']['additionalFees'] > 0) {
            $service = $this->getIdProdutoByCode('-100');
            if ($service) {
                $service = new Item([
                    'idAtendimento' => $this->orderDelivery->idAtendimento,
                    'nrItem' => $this->orderDelivery->items()->count() + 1,
                    'idProduto' => $service['idProduto'],
                    'stOperador' => 'INTEGRACAO',
                    'nrQuantidade' => 1,
                    'dtData' => Carbon::now(),
                    'vrUnitario' => $this->mobyoRequest['total']['additionalFees'],
                    'vrTotal' => $this->mobyoRequest['total']['additionalFees'],
                    'flPrint' => true,
                    'idComanda' => "",
                    'dtAlteracao' => Carbon::now(),
                    'flAtivo' => true,
                    'codigo' => $service['codigo'],
                    'descricao' => 'SERVICO',
                    'cfop' => $service['cfop'],
                    'ncm' => $service['ncm'],
                    'cstIcms' => $service['cstIcms'],
                    'aliquotaIcms' => number_format($service['aliquotaIcms'], 2),
                    'cstPis' => $service['cstPis'],
                    'aliquotaPis' => number_format($service['aliquotaPis'], 2),
                    'cstCofins' => $service['cstCofins'],
                    'aliquotaCofins' => number_format($service['aliquotaCofins'], 2),
                    'ibpt' => number_format($service['ibpt'] / 100 * $this->orderMobyo->getDeliveryFee(), 2),
                    'medida' => $service['medida'],
                ]);
            } else {
                throw new \Exception("Erro ao inserir taxa de serviço: codigo '-100' não cadastrado");
            }
            $this->orderDelivery->items()->save($service);
        }
    }

    private function insertPayments()
    {
        if(!isset($this->mobyo['payments']['methods']) || !$this->mobyo['payments']['methods']) {
            $idFormaPagamento = $this->settingsMobyo->payments[0]->idFormaPagamento;
            $payment = new Payment();
            $payment->setId($this->orderDelivery->idAtendimento);
            $payment->idFormaPagamento = $idFormaPagamento;
            $payment->vrFormaPagamento = $this->orderMobyo->getTotal();
            $payment->dtData = Carbon::now();
            $payment->dtAlteracao = Carbon::now();
            $this->orderDelivery->payments()->save($payment);
            return;
        }
        
        foreach ($this->mobyoRequest['payments']['methods'] as $pay) {
            $idFormaPagamento = 0;
            switch ($this->orderDelivery->engine) {
                case 'EPADOCA':
                    $idFormaPagamento = $this->getIdFormaPagamento($pay['method']);
                    break;
                case 'MOBYO':
                case 'ANOTEAKI':
                    if (isset($pay['method'])) {
                        $idFormaPagamento = $this->getMobyoPayment($pay['method']);
                    } else {
                                                        //PAYMENTS[0] = É A FORMA DE PAGAMENTO 'DEFAULT'
                        $idFormaPagamento = $this->settingsMobyo->payments[0]->idFormaPagamento;
                    }
                    break;
                case 'ZEDELIVERY':
                    $idFormaPagamento = $this->getMobyoPayment($pay['method']);
                    break;
                default:
                                                    //PAYMENTS[0] = É A FORMA DE PAGAMENTO 'DEFAULT'
                    $idFormaPagamento =  $this->settingsMobyo->payments[0]->idFormaPagamento;
                    break;
            }


            if ($this->haveChangefor($pay)) {
                $this->insertPaymentWithChangefor($this->orderDelivery, $pay, $idFormaPagamento);
            } else {
                $payment = new Payment();
                $payment->setId($this->orderDelivery->idAtendimento);
                $payment->idFormaPagamento = $idFormaPagamento;
                $payment->vrFormaPagamento = $this->orderMobyo->getTotal();
                $payment->dtData = Carbon::now();
                $payment->dtAlteracao = Carbon::now();
                $this->orderDelivery->payments()->save($payment);
            }
        }


        if ($this->orderDelivery->payments()->count() == 0)
            throw new \Exception("Não foi possivel inserir o Pagamento");
    }

    private function haveChangefor($pay)
    {
        if(isset($pay['changeFor']) && $pay['changeFor'] > 0) return true;
        return false;
    }

    private function getIdFormaPagamento($code)
    {
        $money = 1;
        $pos = 1;
        foreach($this->settingsMobyo->payments as $item) {
            if($item['name'] == 'money'){
                $money = $item['idFormaPagamento'];
            }
            if($item['name'] == 'pos'){
                $pos = $item['idFormaPagamento'];
            }
        }
        switch ($code) {
            case  '1':
                return $money;
            case '2':
            case '3':
            case '4':
            case '5':
            case '6':
            case '7':
            case '8':
            case '9':
            case '10':
            case '11':
            case '12':
            case '14':
            case '15':
            case '16':
            case '17':
            case '18':
            case '19':
            case '20':
            case '21':
            case '22':
            case '23':
            case '24':
            case '25':
            case '26':
            case '27':
            case '28':
            case '30':
            case '31':
            case '32':
            case '34':
            case '35':
            case '36':
            case '37':
            case '38':
            case '39':
            case '40':
            case '41':
            case '42':
            case '43':
            case '99':
                return $pos;
            case 'DIN':
                return $money;
            case 'CASH':
                return $money;
            case 'CREDIT':
                return $pos;
            case 'DEBIT':
                return $pos;
            default:
                return $pos;
        }
    }

    private function getMobyoPayment($type)
    {
        foreach($this->settingsMobyo->payments as $item) {
            if($item['name'] == strtolower($type)) {
                $payment = $item['idFormaPagamento'];
            }
        }
        if(!isset($payment) || !$payment) {
            $payment = $this->settingsMobyo->payments[0]->idFormaPagamento;
        }
        return $payment;
    }
}
