<?php

namespace App\Services\Financeiro;

use App\Entities\Financeiro\ContaAPagar;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use App\Repositories\ContasAPagarRepository;
use App\Validators\Financeiro\ContaAPagarValidator;
use Prettus\Validator\Contracts\ValidatorInterface;

class ContasAPagarService
{
    private $repository;
    private $validator;
    private $contaBancariaService;
    private $itensContaBancariaService;

    public function __construct(
        ContasAPagarRepository $repository,
        ContaAPagarValidator $validator,
        ContaBancariaService $contaBancariaService,
        ItensContaBancariaService $itensContaBancariaService)
    {
        $this->repository = $repository;
        $this->validator = $validator;
        $this->contaBancariaService = $contaBancariaService;
        $this->itensContaBancariaService = $itensContaBancariaService;
    }

    public function get()
    {
        return $this->repository->all();
    }

    public function getLancamento($request)
    {
        $de = Carbon::createFromFormat('d/m/Y', $request->de)->toDateString();
        $de = $de . ' 00:00:00';
        $ate = Carbon::createFromFormat('d/m/Y', $request->ate)->toDateString();
        $ate = $ate . ' 23:59:59';
        $text = '%' . $request->search . '%';

        if ($request->search) {
            if ($request->per == 'all') {
                $lancamento = ContaAPagar::where('nome_fornecedor', 'LIKE', $text)
                    ->orWhere('conta', 'LIKE', $text)
                    ->orWhere('sub_grupo', 'LIKE', $text)
                    ->orWhere('grupo', 'LIKE', $text)
                    ->orWhere('stDocumento', 'LIKE', $text)
                    ->orWhere('cnpj', 'LIKE', $text)
                    ->orWhere('pagamento', 'LIKE', $text);
            } else {
                $lancamento = ContaAPagar::where($request->per, 'LIKE', $text);
            }
        } else {
            $lancamento = ContaAPagar::query();
        }

        if ($request->chTipo == 'E') {
            $lancamento->where('dtLancamento', '>=', $de)->where('dtLancamento', '<=', $ate)
                ->orderBy('dtLancamento', 'desc');

        } else if ($request->chTipo == 'V') {
            $lancamento->where('dtVencimento', '>=', $de)->where('dtVencimento', '<=', $ate)
                ->orderBy('dtVencimento', 'desc');

        } else if ($request->chTipo == 'P') {
            $lancamento->where('dtLancamento', '>=', $de)->where('dtLancamento', '<=', $ate)
                ->where('flPago', '=', true)
                ->orderBy('dtLancamento', 'desc');
        }

        if ($request->chState == 'T') {
            return $lancamento->get();

        } else if ($request->chState == 'P') {

            return $lancamento->where('flPago', '=', true)->get();

        } else if ($request->chState == 'N') {

            return $lancamento->where('flPago', '=', false)->get();
        }

    }

    public function setBaixa($request, $idContasPagar, $idFornecedor)
    {

        $dtPagamento = Carbon::createFromFormat('d/m/Y', $request['dtPagamento'])->toDateString();
        $lancamento = ContaAPagar::where('idContasPagar', '=', $idContasPagar)->where('idFornecedor', '=', $idFornecedor);
        $vrDocumento = $lancamento->first()->vrDocumento;
        $vrDiferenca = str_replace(',', '.', $request['vrDiferenca']);


        if ($request['stFormato'] == "M") {
            if ($request['stTipoDiferenca'] == "A") {
                $flJuros = 1;
                $vrDocumento = $vrDocumento + $vrDiferenca;
            } else {
                $flJuros = 0;
                $vrDocumento = $vrDocumento - $vrDiferenca;
            }
        } else {
            if ($request['stTipoDiferenca'] == "A") {
                $flJuros = 1;
                $vrDocumento = $vrDocumento + (($vrDiferenca * $vrDocumento) / 100);
            } else {
                $flJuros = 0;
                $vrDocumento = $vrDocumento - (($vrDiferenca * $vrDocumento) / 100);
            }
        }


        return DB::transaction(function () use ($lancamento, $dtPagamento, $vrDocumento, $flJuros, $request) {

            if ($request['isConta'] == false) {
                $idContaBancaria = 0;
            } else {
                $idContaBancaria = $request['idContaBancaria'];
                $lanc = $lancamento->first();

                //Atualiza o saldo da conta bancaria
                $this->contaBancariaService->updateSaldoDeduct($idContaBancaria, $vrDocumento);

                $itensConta = [
                    'idContaBancaria' => $idContaBancaria,
                    'dtData' => Carbon::now(),
                    'stDescricao' => "A PAGAR - " . $request['sub_grupo'] . " [{$request['nome_fornecedor']}]",
                    'stDocumento' => $lanc->stDocumento,
                    'vrDocumento' => $vrDocumento * (-1),
                    'vrSaldoAnterior' => $request['vrSaldoAnterior'],
                    'idContaPagarReceber' => $lanc->idContasPagar,
                    'idFornecedor' => $lanc->idFornecedor,
                    'flCheque' => 0,
                    'dtAlteracao' => Carbon::now(),
                ];
                //itens Conta Bancaria
                $this->itensContaBancariaService->create($itensConta);
            }

            $lancamento->update(['flAtivo' => false,
                'dtPagamento' => $dtPagamento,
                'vrPago' => $vrDocumento,
                'flPago' => 1,
                'flJuros' => $flJuros,
                'stObservacaoBaixa' => isset($request['stObservacaoBaixa']) ? $request['stObservacaoBaixa'] : '',
                'idContaBancaria' => $idContaBancaria]);
        });

    }

    public function selectBaixa($lancamentos)
    {
        $cont = 0;

        foreach ($lancamentos as $lancamento) {

            ContaAPagar::where('idContasPagar', '=', $lancamento['idContasPagar'])
                ->where('idFornecedor', '=', $lancamento['idFornecedor'])->update(['flAtivo' => false,
                    'dtPagamento' => Carbon::now(),
                    'vrPago' => $lancamento['vrDocumento'],
                    'flPago' => 1]);
            $cont++;
        }

        return $cont;
    }

    public function getById($idContasPagar, $idFornecedor)
    {
        $contasReceber = $this->repository->findWhere([
            'idContasPagar' => $idContasPagar,
            'idFornecedor' => $idFornecedor
        ])->first();

        return $contasReceber;
    }


    public function insert($request)
    {
        $contasReceber = $this->data($request);

        $this->validator->with($contasReceber)->passesOrFail(ValidatorInterface::RULE_CREATE);

        return $this->repository->create($contasReceber);
    }

    public function createFromExpense($request)
    {
        return DB::transaction(function () use ($request) {
            $dtLancamento = $request['dtLancamento'];
            $dtVencimento = $request['dtVencimento'];
            $contasReceber = [
                "idContasPagar" => isset($request['idContasPagar']) ? $request['idContasPagar'] : $this->getNextId($request['idFornecedor']) + 1,
                "idFornecedor" => $request['idFornecedor'],
                "idTipoConta" => $request['idSub_grupo'],
                "dtLancamento" => $dtLancamento,
                "dtVencimento" => $dtVencimento,
                "dtPagamento" => isset($request['dtPagamento']) ? $request['dtPagamento'] : null,
                "nrParcela" => 1,
                "nrTotalParcelas" => $request['nrTotalParcelas'],
                "idCentroCusto" => $request['idCentroCusto'],
                "vrDocumento" => str_replace(',', '.', $request['vrFinal']),
                "vrPago" => isset($request['vrPago']) ? $request['vrPago'] : 0.00,
                "vrJuros" => 0.00,
                "flJuros" => null,
                "flPago" => isset($request['flPago']) ? $request['flPago'] : 0,
                "vrDescontos" => isset($request['vrDescontos']) ? $request['vrDescontos'] : 0,
                "vrEncargos" => isset($request['vrEncargos']) ? $request['vrEncargos'] : 0,
                "vrOutros" => isset($request['vrOutros']) ? $request['vrOutros'] : 0,
                "flAtivo" => $request['flAtivo'],
                "stOperador" => $request['stOperador'],
                "stObservacao" => strtoupper($request['stObservacao']),
                "stDocumento" => $request['stDocumento'],
                "stTipoBaixa" => 'I',
                "stTipoJuros" => 'P',
                "stObservacaoBaixa" => isset($request['stObservacaoBaixa']) ? $request['stObservacaoBaixa'] : '',
                "vrDescontoLancamento" => str_replace(',', '.', $request['vrDescontoLancamento']),
                "idContaBancaria" => 0,
                "idConta" => $request['idConta'],
                "dtAlteracao" => Carbon::now(),
                "idTipopagamento" => 1,
            ];

            $this->validator->with($contasReceber)->passesOrFail(ValidatorInterface::RULE_CREATE);

            $res = $this->repository->create($contasReceber);

            $despesa = $request['despesa'];
            $desp = Despesa::where('idDespesa', $despesa['id'])
                ->where('idAbertura', $despesa['id_open'])
                ->update(['flCompensado' => 1, 'dtAlteracao' => Carbon::now()]);
            return $desp;
        });
    }


    public function atualizar($request, $idContasPagar, $idFornecedor)
    {

        $contasReceber = $this->data($request);

        $this->validator->with($contasReceber)->passesOrFail(ValidatorInterface::RULE_UPDATE);

        return ContaAPagar::where('idContasPagar', '=', $idContasPagar)->where('idFornecedor', '=', $idFornecedor)->update($contasReceber);

    }

    public function find($text)
    {

        if (strlen($text) > 0) {
            $text = '%' . $text . '%';
            $lancamento = ContaAPagar::where('stDocumento', 'like', $text)
                ->orWhere('conta', 'like', $text)
                ->orWhere('sub_grupo', 'like', $text)
                ->orWhere('grupo', 'like', $text)
                ->orWhere('nome_fornecedor', 'like', $text)->get();

            return $lancamento;
        }
    }


    public function delete($idContasPagar, $idFornecedor)
    {
        return $this->repository->findWhere([
            'idContasPagar' => $idContasPagar,
            'idFornecedor' => $idFornecedor
        ])->first()->update(['flAtivo' => false]);

    }

    public function getNextId($fornecedor)
    {
        if ($this->repository->findByField('idFornecedor', $fornecedor)->last()) {
            return $idContasPagar = $this->repository->findByField('idFornecedor', $fornecedor)->last()->idContasPagar;
        } else {
            return $idContasPagar = 0;
        }
    }

    public function estorno($idContasPagar, $idFornecedor)
    {
        return DB::transaction(function () use ($idContasPagar, $idFornecedor) {

            $lancamento = $this->getById($idContasPagar, $idFornecedor);

            if ($lancamento->idContaBancaria > 0) {
                $this->contaBancariaService->updateSaldoAdd($lancamento->idContaBancaria, $lancamento->vrPago);

                $itensConta = [
                    'idContaBancaria' => $lancamento->idContaBancaria,
                    'dtData' => Carbon::now(),
                    'stDescricao' => "A PAGAR - " . $lancamento->sub_grupo,
                    'stDocumento' => "ESTORNO - {$lancamento->stDocumento}",
                    'vrDocumento' => $lancamento->vrPago,
                    'vrSaldoAnterior' => 0,
                    'idContaPagarReceber' => $lancamento->idContasPagar,
                    'idFornecedor' => $lancamento->idFornecedor,
                    'flCheque' => 0,
                    'dtAlteracao' => Carbon::now(),
                ];
                //itens Conta Bancaria
                $this->itensContaBancariaService->create($itensConta);
            }
            ContaAPagar::where('idContasPagar', '=', $idContasPagar)->where('idFornecedor', '=', $idFornecedor)
                ->update(['flAtivo' => true, 'dtPagamento' => null, 'flPago' => 0, 'vrPago' => 0.00]);
        });
    }

    public function data($request)
    {

        $dtLancamento = Carbon::createFromFormat('d/m/Y', $request['dtLancamento'])->toDateString();
        $dtVencimento = Carbon::createFromFormat('d/m/Y', $request['dtVencimento'])->toDateString();

        $contasReceber = [
            "idContasPagar" => isset($request['idContasPagar']) ? $request['idContasPagar'] : $this->getNextId($request['idFornecedor']) + 1,
            "idFornecedor" => $request['idFornecedor'],
            "idTipoConta" => $request['idSub_grupo'],
            "dtLancamento" => $dtLancamento,
            "dtVencimento" => $dtVencimento,
            "dtPagamento" => isset($request['dtPagamento']) ? $request['dtPagamento'] : null,
            "nrParcela" => 1,
            "nrTotalParcelas" => $request['nrTotalParcelas'],
            "idCentroCusto" => $request['idCentroCusto'],
            "vrDocumento" => str_replace(',', '.', $request['vrFinal']),
            "vrPago" => isset($request['vrPago']) ? $request['vrPago'] : 0.00,
            "vrJuros" => 0.00,
            "flJuros" => null,
            "flPago" => isset($request['flPago']) ? $request['flPago'] : 0,
            "vrDescontos" => isset($request['vrDescontos']) ? $request['vrDescontos'] : 0,
            "vrEncargos" => isset($request['vrEncargos']) ? $request['vrEncargos'] : 0,
            "vrOutros" => isset($request['vrOutros']) ? $request['vrOutros'] : 0,
            "flAtivo" => $request['flAtivo'],
            "stOperador" => $request['stOperador'],
            "stObservacao" => strtoupper($request['stObservacao']),
            "stDocumento" => $request['stDocumento'],
            "stTipoBaixa" => 'I',
            "stTipoJuros" => 'P',
//            "idFormaPagamento" => $request['idFormaPagamento'],
            "stObservacaoBaixa" => isset($request['stObservacaoBaixa']) ? $request['stObservacaoBaixa'] : '',
            "vrDescontoLancamento" => str_replace(',', '.', $request['vrDescontoLancamento']),
            "idContaBancaria" => 0,
            "idConta" => $request['idConta'],
            "dtAlteracao" => Carbon::now(),
            "idTipopagamento" => 1,
//            "stFormato" => $request['stFormato']
        ];

        return $contasReceber;
    }


}
