<?php

namespace App\Services\Nati;

use App\Services\Service;
use Illuminate\Cache\Events\CacheHit;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;
use App\Entities\Nati\Categoria;
use App\Entities\Nati\SubCategoria;
use App\Repositories\Nati\CategoriaRepository;
use App\Validators\Nati\CategoriaValidator;
use Prettus\Validator\Contracts\ValidatorInterface;
use Carbon\Carbon;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;

class CategoriaService extends Service
{
    private $repository;
    private $validator;
    private $subCategoriaService;

    /**
     * CategoriaService constructor.
     * @param CategoriaRepository $repository
     * @param SubCategoriaService $subCategoriaService
     * @param CategoriaValidator $validator
     */
    public function __construct(CategoriaRepository $repository, SubCategoriaService $subCategoriaService, CategoriaValidator $validator)
    {
        $this->repository = $repository;
        $this->validator = $validator;
        $this->subCategoriaService = $subCategoriaService;
    }

    public function getById($id)
    {
        return $this->repository->with('subCategorias')->withCount('produtos')->find($id);
    }

    public function getAll()
    {
        if (Cache::tags(['ListaCategorias'])->has('all')) {
            return Cache::tags(['ListaCategorias'])->get('all');
        } else {
            $response = $this->repository->with('subCategorias')
                ->orderBy('stCategoria','asc')
                ->findWhere(['flAtivo' => true])
                ;
            Cache::tags(['ListaCategorias'])->put('all', $response, 1440);
            return $response;
        }
    }

    public function all($request)
    {
        $key = md5(json_encode($request));
        if (Cache::tags(['ListaCategorias'])->has($key)) {
            return Cache::tags(['ListaCategorias'])->get($key);
        }
        $order = !isset($request['order']) ? 'ASC' : Str::upper($request['order']);
        $sort = !isset($request['sort']) ? 'idCategoria' : $request['sort'];
        $sort = $sort == 'idCategoria' ? 'prd_Categorias.' . $sort : $sort;

        $result = Categoria::where('prd_Categorias.flAtivo', true)
            ->withCount('produtos')
            ->where(function ($query) use ($request) {
                if (isset($request['filter']) && $request['filter'] != null && strlen($request['filter']) > 0) {
                    $query->where('stCategoria', 'LIKE', '%' . $request['filter'] . '%');
                }
            })
            ->where('flAtivo', true)
            ->orderBy($sort, $order)
            ->paginate($request['limit'], $request['page']);

        Cache::tags(['ListaCategorias'])->put($key, $result, 1440);
        return $result;
    }

    public function create($data)
    {

        try {
            DB::beginTransaction();
            $this->validator->with($data)->passesOrFail(ValidatorInterface::RULE_CREATE);
            $arr = [
                'idCategoria' => $this->getNextId(),
                'stCategoria' => strtoupper($data['stCategoria']),
                'flAtivo' => true,
                'dtAlteracao' => Carbon::now()->toTimeString(),
            ];

            $categoria = $this->repository->create($arr);

            $subCategoria = new SubCategoria();
            $subCategoria->idSubCategoria = 1;
            $subCategoria->idCategoria = $categoria->idCategoria;
            $subCategoria->stSubCategoria = 'SEM SUBCATEGORIA';
            $subCategoria->dtAlteracao = Carbon::now()->toTimeString();
            $subCategoria->save();
            DB::commit();
            return $categoria;

        }catch (\Exception $exception){
            DB::rollBack();
            throw new \Exception($exception->getMessage());
        }

    }

    public function update($request, $id)
    {
        $this->validator->with($request)->passesOrFail(ValidatorInterface::RULE_UPDATE);

        $request['stCategoria'] = Str::upper($request['stCategoria']);
        $result = $this->repository->update($request, $id);
        return $result;
    }

    public function getSubCategorias($id)
    {
        return SubCategoria::where('idCategoria', $id)->get();
    }

    public function deleteSubcategoria($request, $id)
    {
        try {
            $subCategoria = SubCategoria::where('idSubCategoria', $request['id'])
                ->where('idCategoria', $id)->first();

            DB::beginTransaction();
            SubCategoria::where('idSubCategoria', $request['id'])
                ->where('idCategoria', $id)->delete();
            DB::commit();

            return $subCategoria;

        }catch (\Exception $exception){
            DB::rollBack();
        }

    }

    public function addSubcategoria($request, $id)
    {
        if (!isset($request['stSubCategoria'])) {
            throw new ValidationException("O Campo SubCategoria é obrigatorio");
        }

        $categoria = $this->repository->find($id);

        $sub = [
            'idSubCategoria' => $this->subCategoriaService->getNextId($categoria->idCategoria),
            'idCategoria' => $categoria->id,
            'stSubCategoria' => Str::upper($this->cleanString($request['stSubCategoria'])),
            'dtAlteracao' => Carbon::now()->toTimeString()
        ];

        return $categoria->subCategorias()->create($sub);
    }

    public function delete($id)
    {
        $categoria = $this->repository->find($id);

        if ($categoria->produtos->count() > 0) {
            throw new BadRequestHttpException('Não é possivel excluir uma categoria que contém produtos');
        } else {
            $categoria->flAtivo = false;
            $categoria->save();
            return $categoria;
//            return ['deleted' => true];
        }
    }

    public function updateSub($request, $idSub, $id_category)
    {
        return $this->subCategoriaService->update($request, $idSub, $id_category);
    }

    private function getNextId()
    {
        return Categoria::max('idCategoria') + 1;
    }

    public function getOrCreate($categoria)
    {
        $cat = $this->repository->where('stCategoria', strtoupper($categoria))->first();
        if($cat){
            return $cat;
        }else{
            return $this->create(['stCategoria' => strtoupper($categoria)]);
        }
    }
}
