<?php

namespace App\Services\Nati;

use App\Entities\Nati\ImportProducts\ImportProducts;
use App\Entities\Nati\Measure;
use App\Imports\ProductsExport;
use App\Repositories\Nati\ProdutoPerguntaRepository;
use Exception;
use Carbon\Carbon;
use App\Entities\Master;
use App\Services\Service;
use App\Services\Nati\StockBalanceService;
use App\Services\Nati\SubCategoriaService;
use App\Services\Nati\CategoriaService;
use App\Entities\Nati\Code;
use Illuminate\Support\Str;
use App\Entities\Nati\Sector;
use App\Entities\Nati\Product;
use App\Entities\Nati\Aliquota;
use App\Entities\Nati\Categoria;
use App\Entities\Nati\PromocaoLevePague;
use Illuminate\Support\Facades\DB;
use App\Entities\Nati\FichaTecnica;
use App\Entities\Nati\StockBalance;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;
use App\Validators\Nati\ProductValidator;
use App\Repositories\Nati\ProductRepository;
use Maatwebsite\Excel\Facades\Excel;
use Prettus\Validator\Contracts\ValidatorInterface;
use Symfony\Component\HttpFoundation\Exception\BadRequestException;
use App\Exports\Nati\ProductsIntegracaoControllerExport;
use App\Exports\Nati\ProductsExportMobyo;
use App\Exports\Nati\ProductsExportLinx;
use App\Entities\VwNatiProdutos;
use App\Services\Nati\ProdutoPerguntaService;

class ProductService extends Service
{
    protected $repository;
    protected $validator;
    protected $categoriaService;
    protected $aliquotaService;
    protected $setorService;
    protected $medidaService;
    protected $stock;
    protected $nutriService;
    protected $composite;
    protected $fichaFornecedor;
    protected $user;
    protected $master;
    protected $stockBalanceService;
    protected $subCategoriaService;
    protected $importer;
    protected $adjustStocksService;
    private $produtoPerguntaService;

    /**
     * ProdutoService constructor.
     *
     * @param ProductRepository $repository
     * @param ProductValidator $validator
     * @param StockService $stock
     * @param InformationNutritionalService $nutriService
     * @param ProductCompositeService $composite
     * @param FichaFornecedorService $fichaFornecedor
     */
    public function __construct(
        ProductRepository $repository,
        ProductValidator $validator,
        StockService $stock,
        CategoriaService $categoriaService,
        SectorService $setorService,
        MeasureService $medidaService,
        InformationNutritionalService $nutriService,
        AliquotaService $aliquotaService,
        ProductCompositeService $composite,
        FichaFornecedorService $fichaFornecedor,
        StockBalanceService $stockBalanceService,
        AdjustStocksService $adjustStocksService,
        SubCategoriaService $subCategoriaService,
        ImportProducts $importer,
        ProdutoPerguntaService $produtoPerguntaService
    )
    {
        $this->repository = $repository;
        $this->validator = $validator;
        $this->stock = $stock;
        $this->nutriService = $nutriService;
        $this->categoriaService = $categoriaService;
        $this->setorService = $setorService;
        $this->medidaService = $medidaService;
        $this->aliquotaService = $aliquotaService;
        $this->composite = $composite;
        $this->fichaFornecedor = $fichaFornecedor;
        $this->user = auth()->user();
        $this->master = Master::first();
        $this->stockBalanceService = $stockBalanceService;
        $this->adjustStocksService = $adjustStocksService;
        $this->subCategoriaService = $subCategoriaService;
        $this->importer = $importer;
        $this->produtoPerguntaService = $produtoPerguntaService;
    }

    public function get($request)
    {
         $key = md5(json_encode($request));
         if (Cache::tags(['ListaProducts'])->has($key)) {
             return Cache::tags(['ListaProducts'])->get($key);
         } else {
            $order = !$request['order'] ? 'ASC' : Str::upper($request['order']);
            $active = !filter_var($request['inactive'], FILTER_VALIDATE_BOOLEAN);
            if (is_numeric($request['filter'])) {
                $order = 'ASC';
                $request['sort'] = 'codigo';
            }

            switch ($request['sort']) {
                case 'alteracao':
                    $sort = 'alteracao';
                    break;
                case 'categoria':
                    $sort = 'categoria';
                    break;
                case 'codigo':
                    $sort = 'codigo';
                    break;
                case 'descricao':
                    $sort = 'descricao';
                    break;
                case 'id':
                default:
                    $sort = 'id';
                    break;
            }
            $master = $this->master;

            $response = VwNatiProdutos::selectRaw(
                'id,
                codigo,
                descricao,
                descricaoLonga,
                categoria,
                subcategoria,
                medida,
                ncm,
                preco,
                setor,
                aliquota,
                alteracao,
                ativo'
            )
                ->where(function ($query) use ($master, $active) {
                    if (!$master->flPainelAliquotas) {
                        $query->where('ativo', $active);
                    }
                })
                ->where('codigo', '<>', '-100')
                ->where(function ($query) use ($request) {
                    if (isset($request['filter']) && strlen($request['filter']) > 0) {
                        if($request['by'] === 'code') {
                            $search = str_pad($request['filter'], 13, "0", STR_PAD_LEFT);
//                            $query->where('codigo', 'LIKE', "%{$request['filter']}%");
                            $query->where('codigo', '=', $search);
                        } elseif ($request['by'] === 'name') {
//                            $query->where('descricao', 'LIKE', "%{$request['filter']}%");
                            $query->where('descricaoLonga', $request['filter']);
                        } else {
                            $query->where('codigo', 'LIKE', "%{$request['filter']}%")
                                ->orWhere('descricaoLonga', 'LIKE', "%{$request['filter']}%");
                        }

                    }
                })
                ->where('ativo', $active)
                ->orderBy($sort, $order)
                ->paginate($request['limit'], $request['page']);

            Cache::tags(['ListaProducts'])->put($key, $response, 1440);
            return $response;
         }
    }

    public function getPainelAliquotas($request)
    {
        $review = filter_var($request['revision'], FILTER_VALIDATE_BOOLEAN);
        $order = !$request['order'] ? 'ASC' : Str::upper($request['order']);

        return Product::selectRaw(
            'prd_Produtos.idProduto as id,
            MIN(prd_Codigos.idCodigo) as codigo,
            prd_Produtos.stProdutoAbreviado as descricao,
            prd_Produtos.vrUnitario as preco,
            prd_Produtos.idAliquota,
            prd_Produtos.nrNCm as ncm,
            prd_Produtos.stCest as cest,
            prd_Produtos.flRevisaoTributaria as revisao,
            prd_Produtos.dtRevisaoTributaria as dataRevisao,
            prd_Produtos.stOperadorRevisaoTributaria as usuarioRevisao'
        )
            ->join('prd_Codigos', 'prd_produtos.idProduto', '=', 'prd_Codigos.idProduto')
            ->join('prd_Aliquotas', 'prd_Produtos.idAliquota', '=', 'prd_Aliquotas.idAliquota')
            ->join('prd_Setores', 'prd_Produtos.idSetor', '=', 'prd_Setores.idSetor')
            ->join('prd_Categorias', 'prd_Produtos.idCategoria', '=', 'prd_Categorias.idCategoria')
            ->where('prd_Codigos.idCodigo', '<>', '-100')
            ->where(function ($query) use ($review) {
                if ($review) {
                    $query->where('prd_Produtos.flRevisaoTributaria', false);
                }
            })
            ->where(function ($query) use ($request) {
                if (isset($request['filter']) && strlen($request['filter']) > 0) {
                    $query->where('prd_Codigos.idCodigo', $request['filter'])
                        ->orWhere('prd_Produtos.stProdutoAbreviado', 'LIKE', "%{$request['filter']}%")
                        ->orWhere('prd_Produtos.nrNcm', $request['filter']);
                }
            })
            ->where(function ($query) use ($request) {
                if ($request['setor'] != null && strlen($request['setor']) > 0)
                    $query->where('prd_Produtos.idSetor', $request['setor']);

                if ($request['aliquota'] != null && strlen($request['aliquota']) > 0)
                    $query->where('prd_Produtos.idAliquota', $request['aliquota']);

                if ($request['categoria'] != null && strlen($request['categoria']) > 0)
                    $query->where('prd_Produtos.idCategoria', $request['categoria']);
            })
            ->orderBy('prd_Produtos.dtAlteracao', $order)
            ->groupBy(['prd_Produtos.idProduto',
                'prd_Produtos.stProdutoAbreviado',
                'prd_Produtos.vrUnitario',
                'prd_Produtos.idAliquota',
                'prd_Produtos.nrNCm',
                'prd_Produtos.stCest',
                'prd_Produtos.flRevisaoTributaria',
                'prd_Produtos.dtRevisaoTributaria',
                'prd_Produtos.stOperadorRevisaoTributaria',
                'prd_Produtos.dtAlteracao'])
            ->paginate($request['limit'], $request['page']);
    }

    public function getById($id)
    {
        $result = $this->repository->with(['composites', 'nutritional', 'stock', 'medida', 'perguntas', 'codigos', 'promoLevePague'])->find($id);
        $result->dtInicioPromocaoHora = Carbon::parse( $result->dtInicioPromocaoHora)->format('H:i');
        $result->dtFimPromocaoHora = Carbon::parse( $result->dtFimPromocaoHora)->format('H:i');
        $result->dtInicioPromocao = Carbon::parse($result->dtInicioPromocao)->format('d/m/Y H:m');
        $result->dtFimPromocao = Carbon::parse($result->dtFimPromocao)->format('d/m/Y H:m');
        $result->nutritional = $result->nutritional->toArray();
        if($result->promoLevePague) {
            $result->promoLevePague->dtInicioPromocao = Carbon::parse($result->promoLevePague->dtInicioPromocao)->format('d/m/Y');
            $result->promoLevePague->dtFimPromocao = Carbon::parse($result->promoLevePague->dtFimPromocao)->format('d/m/Y');;
        }
        return $result;
    }

    private function getNextId()
    {
        return Product::max('idProduto') + 1;
    }

    private function getNextEstSaldoId()
    {
        return StockBalance::max('idSaldo') + 1;
    }

    public function verifyCode($id)
    {
        $code = Code::where('idCodigo', $id)->get();

        return response()->json(count($code) ? false : true);
    }

    public function filter($data)
    {
        return Product::select([
            'prd_produtos.idProduto AS id',
            'stProdutoAbreviado AS descricao',
            'vrUnitario AS preco',
            DB::raw('MIN(idCodigo) AS codigo')
        ])
            ->join('prd_codigos', 'prd_produtos.idProduto', '=', 'prd_Codigos.idProduto')
            ->where('prd_Produtos.flAtivo', true)
            ->where(function ($query) use ($data) {
                $query->where('prd_Codigos.idCodigo', "$data")
                    ->orWhere('prd_Produtos.stProdutoAbreviado', 'LIKE', "%$data%");
            })
            ->where('prd_Codigos.idCodigo', '<>', '-100')
            ->groupBy('prd_produtos.idProduto', 'prd_produtos.stProdutoAbreviado', 'prd_produtos.vrUnitario')
            ->paginate(10);
    }

    public function export()
    {
        return Excel::download(new ProductsExport, 'produtos.xlsx');
    }

    public function exportForIntegration()
    {
        return $response = Excel::download(new ProductsIntegracaoControllerExport, 'produtos.csv');
    }

    public function exportForMobyo()
    {
        return Excel::download(new ProductsExportMobyo, 'produtos_mobyo.xlsx');
    }
    public function exportForLinx()
    {
        return Excel::download(new ProductsExportLinx, 'produtos.xlsx');
    }

    public function import($request)
    {
        try {
            $pathFile = $request->file('file')->getRealPath();
//            $importer = New ImportProducts();

            return $this->importer->importFile($pathFile);
        } catch (Exception $exception) {
            DB::rollBack();
          throw new Exception($exception->getMessage());
        }
    }

//    public function import($data)
//    {
//        try {
//            DB::beginTransaction();
//
//            $product = $this->parseCsv($data);
//            $this->validator->with($product)->passesOrFail(ValidatorInterface::RULE_CREATE);
//            $product = $this->repository->create($product);
//
//
//            Code::where('idProduto', $product->idProduto)->delete();
//
//            if (!isset($data['Codigo']) && !isset($data['EAN'])) {
//                throw new Exception("Nenhum codigo foi informado para o produto");
//            }
//            if (isset($data['Codigo']) && Str::length($data['Codigo']) > 0) {
//                $data['Codigo'] = $this->trataCodigo($data['Codigo']);
//                Code::create(['idCodigo' => $data['Codigo'], 'idProduto' => $product->idProduto]);
//            }
//            if (isset($data['EAN']) && Str::length($data['EAN'])) {
//                $data['EAN'] = $this->trataCodigo($data['EAN']);
//                Code::create(['idCodigo' => $data['EAN'], 'idProduto' => $product->idProduto]);
//            }
//
//            DB::commit();
//        } catch (Exception $exception) {
//            DB::rollBack();
//        }
//    }

    public function parseCsv($data)
    {
        $data['CST-ICMS'] = str_pad(utf8_encode($data['CST-ICMS']), 2, '0', STR_PAD_LEFT);
        $data['CST-PIS'] = str_pad($data['CST-PIS'], 2, '0', STR_PAD_LEFT);
        $data['CST-COFINS'] = str_pad($data['CST-COFINS'], 2, '0', STR_PAD_LEFT);

        $categoria = $this->categoriaService->getOrCreate((empty($data['Categoria'])) ? $data['Categoria'] = 'SEM CATEGORIA' : $this->trataString(utf8_encode($data['Categoria'])));
        $setor = $this->setorService->getOrCreate((empty($data['Setor'])) ? $data['Setor'] = 'SEM SETOR' : $this->trataString(utf8_encode($data['Setor'])));
        $medida = $this->medidaService->getOrCreate((empty($data['Medida'])) ? $data['Medida'] = 'UN' : $this->trataString(utf8_encode($data['Medida'])));
        $aliquota = $this->aliquotaService->getOrCreate($data);
        $subCategoria = $this->subCategoriaService->getOrCreate((empty($data['Subcategoria'])) ? $data['Subcategoria'] = 'SEM SUBCATEGORIA' : $this->trataString(utf8_encode($data['Subcategoria'])), $categoria->idCategoria);

        $product = [];
        $product['idProduto'] = isset($data['idProduto']) ? $data['idProduto'] : $this->getNextId();
        $product['stProduto'] = $this->trataString(utf8_encode(strtoupper($data['Descricao'])));
        $product['stProdutoAbreviado'] = $this->trataString(utf8_encode(strtoupper(Str::substr($data['Descricao'], 0, 29))));
        $product['stOperador'] = auth()->user()->stApelido;
        $product['idCategoria'] = $categoria->idCategoria;
        $product['idSetor'] = $setor->idSetor;
        $product['idMedida'] = $medida->idMedida;

        $product['vrUnitario'] = str_replace(',','.', $data['Preco']);
        $product['vrCusto'] = empty($data['Custo']) ? $data['Custo'] = 0 : str_replace(',', '.', $data['Custo']);
        $product['vrMargemLucro'] = empty($data['Margem']) ? $data['vrMargemLucro'] = 0 : $data['Margem'];
        $product['vrUnitarioPromocao'] = str_replace(',','.', $data['Preco']);
        $product['vrUnitarioPromocaoHora'] = str_replace(',','.', $data['Preco']);
        $product['vrUnitario2'] = str_replace(',','.', $data['Preco']);
        $product['idTipoProduto'] = 1;

        $product['stCest'] = isset($data['CEST']) ? $data['CEST'] : null;
        $product['nrNCM'] = empty($data['NCM']) ? $data['NCM'] = '19059090' : str_replace('.', '', $data['NCM']);

        $product['idAliquota'] = $aliquota->idAliquota;

        $product['flComissao'] = true;
        $product['idLocalizacao'] = 1;
        $product['idSubCategoria'] = $subCategoria['idSubCategoria'];
        $product['dtUltimaAlteracao'] = Carbon::now();
        $product['dtAlteracao'] = Carbon::now();
        $product['dtInicioPromocao'] = '2008-01-01 00:00:00.000';
        $product['dtFimPromocao'] = '2008-01-01 00:00:00.000';
        $product['dtInicioPromocaoHora'] = '2008-01-01 00:00:00.000';
        $product['dtFimPromocaoHora'] = '2008-01-01 00:00:00.000';
        $product['flPocket'] = true;
        $product['nr10Mais'] = 0;
        $product['flPromocao'] = false;
        $product['nrValidade'] = 0;
        $product['flControlarEstoque'] = false;

        $product['flVendaCartao'] = true;
        $product['flVendaTicket'] = false;
        $product['flVendaSmart'] = false;
        $product['flRequererItemAdicional'] = false;
        $product['nrFatorConversao'] = 0;
        $product['nrTipoProduto'] = 0;
        $product['idProdutoVinculado'] = 0;
        $product['flCupomPreVenda'] = false;
        $product['flPromocaoHora'] = false;
        $product['stDescricao'] = '';
        $product['flAtivo'] = true;

        return $product;
    }

    private function trataString($string)
    {
        $comAcentos = array('à', 'á', 'â', 'ã', 'ä', 'å', 'ç', 'è', 'é', 'ê', 'ë', 'ì', 'í', 'î', 'ï', 'ñ', 'ò', 'ó', 'ô', 'õ', 'ö', 'ù', 'ü', 'ú', 'ÿ', 'À', 'Á', 'Â', 'Ã', 'Ä', 'Å', 'Ç', 'È', 'É', 'Ê', 'Ë', 'Ì', 'Í', 'Î', 'Ï', 'Ñ', 'Ò', 'Ó', 'Ô', 'Õ', 'Ö', 'Ù', 'Ü', 'Ú', '-', '/', '.', ',');
        $semAcentos = array('a', 'a', 'a', 'a', 'a', 'a', 'c', 'e', 'e', 'e', 'e', 'i', 'i', 'i', 'i', 'n', 'o', 'o', 'o', 'o', 'o', 'u', 'u', 'u', 'y', 'A', 'A', 'A', 'A', 'A', 'A', 'C', 'E', 'E', 'E', 'E', 'I', 'I', 'I', 'I', 'N', 'O', 'O', 'O', 'O', 'O', 'U', 'U', 'U', '', '', '', '');

        return str_replace($comAcentos, $semAcentos, $string);
    }



    public function create($data)
    {
        try {
            $product = $this->mapToEntity($data, true);

            $this->validator->with($product)->passesOrFail(ValidatorInterface::RULE_CREATE);

            return DB::transaction(function () use ($data, $product) {
                $product = $this->repository->create($product);
                $this->setCode($data, $product->idProduto);
                $this->setStock($data['flControlarEstoque'], $data['stock'], $product, true);
                if (isset($data['nutritional']) && isset($data['nutritional'][0])) {
                    $x = $product->idProduto;
                    $this->setInfoNutritional($data['nutritional'], $x);
                }
                if (isset($data['flRequererItemAdicional'])) {
                    $this->setComposite($data['flRequererItemAdicional'], $data['composites'], $product['idProduto']);
                }
                if($data['perguntas']) {
                    $data['idProduto'] = $product->idProduto;
                    $this->linkProductQuestion($data);
                }
                if (!empty($data['promo_leve_pague'])) {
                    try {
                        $promoData = $data['promo_leve_pague'];
    
                        $dtInicioPromocao = isset($promoData['dtInicioPromocao']) 
                            ? \Carbon\Carbon::createFromFormat('d/m/Y', $promoData['dtInicioPromocao'])->format('Y-m-d') 
                            : null;
    
                        $dtFimPromocao = isset($promoData['dtFimPromocao']) 
                            ? \Carbon\Carbon::createFromFormat('d/m/Y', $promoData['dtFimPromocao'])->format('Y-m-d') 
                            : null;
    
                        PromocaoLevePague::create([
                            'idProduto' => $product->idProduto,
                            'nrLeve' => $promoData['nrLeve'],
                            'nrPague' => $promoData['nrPague'],
                            'dtInicioPromocao' => $dtInicioPromocao,
                            'dtFimPromocao' => $dtFimPromocao,
                            'hrInicioPromocao' => $promoData['hrInicioPromocao'],
                            'hrFimPromocao' => $promoData['hrFimPromocao'],
                            'dias' => $promoData['dias'],
                        ]);
                    } catch (Exception $e) {
                        throw new Exception("Erro ao criar a promoção leve-pague: " . $e->getMessage());
                    }
                }
                return $product;
            });
        } catch (Exception $exception) {
            throw new Exception($exception->getMessage());
        }

    }

    public function update($data, $id)
    {
        $product = $this->mapToEntity($data);
        $this->validator->with($product)->passesOrFail(ValidatorInterface::RULE_UPDATE);

        if($data['perguntas']) {
            try {
                $this->linkProductQuestion($data);
            } catch (Exception $e) {
                throw new Exception($e->getMessage());
            }
        }

        if($data['perguntasRemovidas']) {
            try {
                $this->unlinkProductQuestion($data);
            } catch (Exception $e) {
                throw new Exception($e->getMessage());
            }
        }

        if (!empty($data['promo_leve_pague'])) {
            try {
                $promoData = $data['promo_leve_pague'];
                $promo = PromocaoLevePague::where('idProduto', $id)->first();

                $dtInicioPromocao = isset($promoData['dtInicioPromocao']) ? \Carbon\Carbon::createFromFormat('d/m/Y', $promoData['dtInicioPromocao'])->format('Y-m-d') : null;
                $dtFimPromocao = isset($promoData['dtFimPromocao']) ? \Carbon\Carbon::createFromFormat('d/m/Y', $promoData['dtFimPromocao'])->format('Y-m-d') : null;
                
                if($promo) {
                    $promo->update([
                        'nrLeve' => $promoData['nrLeve'] ?? $promo->nrLeve,
                        'nrPague' => $promoData['nrPague'] ?? $promo->nrPague,
                        'dtInicioPromocao' => $dtInicioPromocao ?? $promo->dtInicioPromocao,
                        'dtFimPromocao' => $dtFimPromocao ?? $promo->dtFimPromocao,
                        'hrInicioPromocao' => $promoData['hrInicioPromocao'] ?? $promo->hrInicioPromocao,
                        'hrFimPromocao' => $promoData['hrFimPromocao'] ?? $promo->hrFimPromocao,
                        'dias' => $promoData['dias'] ?? $promo->dias,
                    ]);
                } else {
                    PromocaoLevePague::create([
                        'idProduto' => $id,
                        'nrLeve' => $promoData['nrLeve'],
                        'nrPague' => $promoData['nrPague'],
                        'dtInicioPromocao' => $dtInicioPromocao,
                        'dtFimPromocao' => $dtFimPromocao,
                        'hrInicioPromocao' => $promoData['hrInicioPromocao'],
                        'hrFimPromocao' => $promoData['hrFimPromocao'],
                        'dias' => $promoData['dias'],
                    ]);
                }
            } catch (Exception $e) {
                throw new Exception("Erro ao salvar promoção leve-pague: " . $e->getMessage());
            }
        }

        try {
            return DB::transaction(function () use ($product, $data, $id) {
                $this->setCode($data, $product['idProduto']);
                if(isset($data['flControlarEstoque']) && $data['flControlarEstoque'] == true){
                    foreach($data['stock'] as  $stock) {
                        $amountStock = $stock['nrQuantidade'] + $stock['newQuantity'];
                        $this->setStock($data['flControlarEstoque'], $amountStock, $product, $stock['idEstoque'], false);
                    }
                }
                if (isset($data['nutritional'])) {
                    $this->setUpdateInfoNutritional($data['nutritional'], $data['nutritional'], $product['idProduto']);
                }
                if(isset($data['flRequererItemAdicional'])){
                    $this->setComposite($data['flRequererItemAdicional'], $data['composites'], $product['idProduto']);
                }
                return $this->repository->update($product, $id);
            });
        } catch (Exception $e) {
            throw new Exception($e->getMessage());
        }
    }

    public function updateTaxData($id, $request)
    {
        $product = Product::find($id);

        if ($product->dtRevisaoTributaria == null) {
            return $product->update([
                'idAliquota' => $request['idAliquota'],
                'stCest' => $request['cest'],
                'nrNCM' => $request['ncm'],
                'flRevisaoTributaria' => true,
                'flAtivo' => true,
                'dtUltimaAlteracao' => Carbon::now()->toDateTimeString(),
                'dtRevisaoTributaria' => Carbon::now()->toDateTimeString(),
                'stOperadorRevisaoTributaria' => auth()->user()->stApelido
            ]);
        } else {
            return $product->update([
                'idAliquota' => $request['idAliquota'],
                'stCest' => $request['cest'],
                'nrNCM' => $request['ncm'],
                'flRevisaoTributaria' => true,
                'flAtivo' => true,
                'dtUltimaAlteracao' => Carbon::now()->toDateTimeString()
            ]);
        }
    }

    public function updateTaxDatas(array $request)
    {
        foreach ($request['products'] as $product) {

            $prod = Product::find($product['id']);
            $apelido = Auth::user()->stApelido;

            $prod->update([
                'idAliquota' => $product['idAliquota'],
                'stCest' => $product['cest'],
                'nrNCM' => $product['ncm'],
                'flRevisaoTributaria' => true,
                'flAtivo' => true,
                'dtUltimaAlteracao' => Carbon::now()->toDateTimeString(),
                'dtRevisaoTributaria' => Carbon::now()->toDateTimeString(),
                'stOperadorRevisaoTributaria' => $apelido
            ]);

        }

        return true;
    }

    public function delete($id)
    {
        $this->produtoPerguntaService->deleteByProduct($id);
    //    Code::where('idProduto', $id)->update(['idCodigo' => 'SEM COD']);
        return Product::find($id)->update([
            'flAtivo' => false,
            'dtUltimaAlteracao' => Carbon::now()->toTimeString(),
            'dtAlteracao' => Carbon::now()->toTimeString()
        ]);
    }

    public function active($id)
    {
        Product::find($id)->update(['flAtivo' => true, 'dtUltimaAlteracao' => Carbon::now()]);

        return Product::find($id);
    }

    public function activeStock($id)
    {
        return DB::transaction(function () use ($id) {
            $prod = Product::find($id);
            $this->setStock(true, 0, $prod->toArray());
            $prod->flControlarEstoque = true;
            $prod->dtUltimaAlteracao = Carbon::now()->toTimeString();
            $prod->save();
        });
    }

    public function updatedInventoryControl($id)
    {
        return Product::find($id)->update(['flControlarEstoque' => true, 'dtUltimaAlteracao' => Carbon::now()->toTimeString()]);
    }

    public function updatedCost($id, $cost)
    {
        $product = Product::find($id);

        if ($product['vrCusto'] != $cost) {
            return $product->update(['vrCusto' => $cost, 'dtUltimaAlteracao' => Carbon::now()->toTimeString()]);
        }
    }

    public function destroy($id)
    {
        $this->produtoPerguntaService->deleteByProduct($id);

        return DB::transaction(function () use ($id) {
            // Code::where('idProduto', $id)->delete();
             $this->repository->update([
                 'flAtivo' => false,
                 'dtUltimaAlteracao' => Carbon::now()->toTimeString(),
                 'dtAlteracao' => Carbon::now()->toTimeString()
             ], $id);
            return $this->repository->find($id);
        });
    }
    public function linkProductQuestion($data)
    {
        foreach ($data['perguntas'] as $question) {
            if(!$data['idProduto']) return;
            if(!$question['idPergunta']) return;
            $result = $this->produtoPerguntaService->linkProductQuestion($question, $data['idProduto']);
        }
    }
    public function unlinkProductQuestion($data)
    {
        foreach ($data['perguntasRemovidas'] as $removedQuestion) {
            if(!$removedQuestion['idPergunta']) return;
            $result = $this->produtoPerguntaService->unlinkProductQuestion($data['idProduto'], $removedQuestion['idPergunta']);
        }
    }

    public function setCode($data, $id_produtct)
    {
        if (!$data['codes']) {
            throw new BadRequestException('Nenhum Codigo foi informado');
        }
        Code::where('idProduto', $id_produtct)->delete();
        foreach ($data['codes'] as $code) {
            Code::create(['idCodigo' => $code, 'idProduto' => $id_produtct]);
        }
    }

    public function setStock($inventory_control, $new_quantity, $product, $stockId = 1, $create = false )
    {
        $data['idProduto'] = $product['idProduto'];
        $data['idEstoque'] = $stockId;
        $data['newQuantity'] = $new_quantity;
        $data['nrQuantidade'] = 0;
        if ($inventory_control && $create) {
            $balance = StockBalance::where('idProduto', $product['idProduto'])
                                    ->where('idEstoque', $data['idEstoque'])
                                    ->first();
            if($balance) {
                $data['nrQuantidade'] = $balance->nrQuantidade;
                $this->adjustStocksService->removeIfDuplicated($product['idProduto']);
                $balance->nrQuantidade = $new_quantity;
                $balance->dtModificacao = Carbon::now()->toTimeString();
                $balance->dtConferencia = Carbon::now()->toTimeString();
                $balance->dtAlteracao = Carbon::now()->toTimeString();
                $balance->save();
                return;
            } else {
                $stock = StockBalance::create([
                    'idSaldo' => $this->getNextEstSaldoId(),
                    'idEstoque' => $data['idEstoque'],
                    'idProduto' => $product['idProduto'],
                    'nrQuantidade' => $new_quantity,
                    'nrReservas' => 0,
                    'dtModificacao' => Carbon::now()->toTimeString(),
                    'dtConferencia' => Carbon::now()->toTimeString(),
                    'nrMinimo' => 0,
                    'dtAlteracao' => Carbon::now()->toTimeString(),
                ]);
                $stock->save();
                $this->adjustStocksService->insertMovimentation($data);
                return;
            }
        } else if ($inventory_control) {

            $prod = Product::find($product['idProduto']);

            if ($prod->flControlarEstoque == false) {
                $stock = StockBalance::where('idProduto', $product['idProduto'])->where('idEstoque', $data['idEstoque'])->first();

                if ($stock) {
                    $stock->nrQuantidade = $new_quantity;
                    $stock->save();


                } else {
                    $stock = StockBalance::create([
                        'idSaldo' => $this->getNextEstSaldoId(),
                        'idEstoque' => $data['idEstoque'],
                        'idProduto' => $product['idProduto'],
                        'nrQuantidade' => $new_quantity,
                        'nrReservas' => 0,
                        'dtModificacao' => Carbon::now()->toTimeString(),
                        'dtConferencia' => Carbon::now()->toTimeString(),
                        'nrMinimo' => 0,
                        'dtAlteracao' => Carbon::now()->toTimeString(),
                    ]);
                    $stock->save();

                    $this->adjustStocksService->insertMovimentation($data);
                }
            }

            if ($prod->flControlarEstoque) {
                $stock = StockBalance::where('idProduto', $product['idProduto'])->where('idEstoque', $stockId)->first();
                if ($stock) {
                    $this->adjustStocksService->removeIfDuplicated($product['idProduto']);
                    DB::table('est_saldos')
                                ->where('idSaldo', $stock->idSaldo)
                                ->update(['nrQuantidade' => $new_quantity,
                                          'dtModificacao' => Carbon::now(),
                                          'dtConferencia' => Carbon::now(),
                                          'dtAlteracao' => Carbon::now()]);
                                          
                    $this->adjustStocksService->insertMovimentation($data);

                } else {
                    $stock = StockBalance::create([
                        'idSaldo' => $this->getNextEstSaldoId(),
                        'idEstoque' => $data['idEstoque'],
                        'idProduto' => $product['idProduto'],
                        'nrQuantidade' => $new_quantity,
                        'nrReservas' => 0,
                        'dtModificacao' => Carbon::now()->toTimeString(),
                        'dtConferencia' => Carbon::now()->toTimeString(),
                        'nrMinimo' => 0,
                        'dtAlteracao' => Carbon::now()->toTimeString(),
                    ]);
                    $stock->save();
                }
            }
        }
    }


    public function setInfoNutritional($data, $id_product)
    {
        return $this->nutriService->insert($data, $id_product);
    }

    public function setUpdateInfoNutritional($nutritional_information, $nutriRequest, $id_product)
    {
        $nutritionals = $this->nutriService->getById($id_product);
        if ($nutritional_information == true) {
            if ($nutritionals) {
                foreach($nutriRequest as $it) {
                    if(isset($it['idTabelaNutricional'])) {
                        $this->nutriService->update($it);
                    } else {
                        $this->nutriService->insert($it, $id_product);
                    }
                }
            } else {
                foreach($nutriRequest as $nutri) {
                    $this->nutriService->insert($nutri, $id_product);
                }
                return;
            }
        } else {
            if ($nutritionals) {
                return $this->nutriService->deleteByIdProduct($id_product);
            }
        }
    }

    public function setComposite($composite, $data, $id_product)
    {
        $this->composite->delete($id_product);

        if ($composite == true) {
            return $this->composite->create($id_product, $data);
        }
    }

    public function mapToEntity($data, $create = false)
    {
        $result = [
            'idProduto' => isset($data['idProduto']) ? $data['idProduto'] : $this->getNextId(),
            'stProduto' => $data['stProduto'],
            'stProdutoAbreviado' => $data['stProdutoAbreviado'],
            'stOperador' => auth()->user()->stApelido,
            'idCategoria' => $data['idCategoria'],
            'idSubCategoria' => $data['idSubCategoria'],
            'idSetor' => $data['idSetor'],
            'idMedida' => $data['idMedida'],
            'idLocalizacao' => isset($data['idLocalizacao']) ? $data['idLocalizacao'] : 1,
            'idAliquota' => $data['idAliquota'],
            'vrUnitario' => $data['vrUnitario'],
            'vrCusto' => $data['vrCusto'],
            'flComissao' => $data['flComissao'],
            'nrValidade' => isset($data['nrValidade']) ? $data['nrValidade'] : 0,
            'flControlarEstoque' => $data['flControlarEstoque'],
            'idTipoProduto' => $data['idTipoProduto'],
            'flPocket' => true,
            'nr10Mais' => 0,
            'flPromocao' => $data['flPromocao'],
            'flBalanca' => $data['flBalanca'],
            'dtUltimaAlteracao' => Carbon::now()->toDateTimeString(),
            'dtAlteracao' => Carbon::now()->toTimeString(),
            'dtInicioPromocao' => $data['flPromocao'] == true ?
                Carbon::parse(str_replace('/', '-', $data['dtInicioPromocao']))
                    ->toDateTimeString() : '2008-01-01 00:00:00.000',
            'dtFimPromocao' => $data['flPromocao'] == true ?
                Carbon::parse(str_replace('/', '-', $data['dtFimPromocao']))
                    ->toDateTimeString() : '2008-01-01 00:00:00.000',
            'dtInicioPromocaoHora' => (isset($data['flPromocaoHora']) && $data['flPromocaoHora'] == true) ?
                    Carbon::parse('2008-01-01 '.$data['dtInicioPromocaoHora'])
                        ->toDateTimeString() : '2008-01-01 00:00:00.000',
            'dtFimPromocaoHora' => (isset($data['flPromocaoHora']) && $data['flPromocaoHora'] == true) ?
                    Carbon::parse('2008-01-01 '.$data['dtFimPromocaoHora'])
                        ->toDateTimeString() : '2008-01-01 00:00:00.000',
            'vrUnitarioPromocao' => isset($data['vrUnitarioPromocao']) ? $data['vrUnitarioPromocao'] : 0,
            'flVendaCartao' => $data['flVendaCartao'],
            'flVendaTicket' => false,
            'flVendaSmart' => false,
            'flRequererItemAdicional' => isset($data['flRequererItemAdicional']) ? $data['flRequererItemAdicional'] : false,
            'vrMargemLucro' => $data['vrMargemLucro'],
            'nrFatorConversao' => 0,
            'stDescricao' => $data['stDescricao'],
            'nrTipoProduto' => 0,
            'idProdutoVinculado' => 0,
            'flCupomPreVenda' => $data['flCupomPreVenda'],
            'nrNCM' => $data['nrNCM'],
            'flPromocaoHora' => isset($data['flPromocaoHora']) and $data['flPromocaoHora'] == true ? $data['flPromocaoHora'] : false,
            'flPromocaoLevePague' => isset($data['flPromocaoLevePague']) ? $data['flPromocaoLevePague'] : false,
            'vrUnitarioPromocaoHora' => isset($data['vrUnitarioPromocaoHora']) ? $data['vrUnitarioPromocaoHora'] : 0,
            'stCest' => isset($data['stCest']) ? $data['stCest'] : null,
            'vrUnitario2' => isset($data['vrUnitario2']) ? $data['vrUnitario2'] : 0,
            'flAtivo' => true,
            'flDesconto' => isset($data['flDesconto']) ? $data['flDesconto'] : 1
        ];

//        if ($create && $this->master->flPainelAliquotas)
//            $result['flAtivo'] = false;

        return $result;
    }

    public function nextcode()
    {
        $qry = "SELECT MIN(CONVERT(bigint, idCodigo)) + 1 AS codigo FROM prd_Codigos c
	            WHERE not exists (SELECT * FROM prd_Codigos c2 WHERE CONVERT(bigint, c.idCodigo) + 1 = c2.idCodigo)
	            AND (CONVERT(bigint, c.idCodigo)) > 0";
        $result = DB::select($qry);
        return $result[0]->codigo;
    }

    public function getByGroup($id, $group)
    {
        switch ($group) {
            case 'aliquota':
                return Aliquota::findOrFail($id)->produtos()->with('codigos')->get();
            case 'categoria':
                return Categoria::findOrFail($id)->produtos()->with('codigos')->get();
            case 'setor':
                return Sector::findOrFail($id)->produtos()->with('codigos')->get();
            default:
                throw new BadRequestException("Grupo Inválido");
        }
    }

    public function updatePrice($produtos)
    {
        $cont = 0;

        foreach ($produtos as $produto) {

            if ($produto['novoValor'] != 0) {

                $this->repository->update(['vrUnitario' => $produto['novoValor']], $produto['idProduto']);
                $cont++;
            }
        }

        return "Preço de " . $cont . " produtos atualizados.";
    }

    public function updateNcm($ncms)
    {
        $cont = 0;
        foreach ($ncms as $ncm) {

            $this->repository->update(['nrNCM' => $ncm['nrNCM']], $ncm['idProduto']);
            $cont++;
        }
        return "NCM de " . $cont . " produtos atualizados.";
    }

    public function getPrintConfig($request)
    {
        $products = Product::selectRaw(
            '[prd_Produtos].[idProduto],
            MIN([prd_Codigos].[idCodigo]) as idCodigo,
            [prd_Produtos].[stProdutoAbreviado],
            [prd_Categorias].[stCategoria]'
        )
            ->join('prd_Setores', 'prd_Produtos.idSetor', '=', 'prd_Setores.idSetor')
            ->join('prd_Categorias', 'prd_Produtos.idCategoria', '=', 'prd_Categorias.idCategoria')
            ->join('prd_Codigos', 'prd_Produtos.idProduto', '=', 'prd_Codigos.idProduto')
            ->where('prd_Produtos.flAtivo', true)
            ->where('prd_Codigos.idCodigo', '<>', '-100')
            ->where('prd_Produtos.idSetor', $request['sector'])
            ->groupBy('prd_Produtos.idProduto', 'prd_Produtos.stProdutoAbreviado', 'prd_Categorias.stCategoria')
            ->orderBy('prd_Categorias.stCategoria')
            ->orderBy('prd_Produtos.stProdutoAbreviado')
            ->get();

        $config = FichaTecnica::where(['idPrint' => $request['printer'], 'idMicroterminal' => $request['terminal'], 'idSetor' => $request['sector']])->get()->map(function ($item) {
            return $item->idProduto;
        })->toArray();

        return $products->mapToGroups(function ($product) use ($config) {
            return [
                $product['stCategoria'] => [
                    'id' => $product['idProduto'],
                    'code' => $product['idCodigo'],
                    'description' => $product['stProdutoAbreviado'],
                    'active' => in_array($product['idProduto'], $config)
                ]
            ];
        });
    }

    public function moveCategory($category, $sub_category, $ids)
    {
        $len = count($ids);

        if ($len > 2100) {
            $ids_chunk = array_chunk($ids, 1000);
            foreach ($ids_chunk as $partition_id) {
                Product::whereIn('idProduto', $partition_id)->update([
                    'idCategoria' => $category,
                    'idSubCategoria' => $sub_category,
                    'dtUltimaAlteracao' => Carbon::now()->toDateTimeString()
                ]);
            }
        } else {
            Product::whereIn('idProduto', $ids)->update([
                'idCategoria' => $category,
                'idSubCategoria' => $sub_category,
                'dtUltimaAlteracao' => Carbon::now()->toDateTimeString()
            ]);
        }
        return [
            'updated' => true
        ];
    }

    public function moveSector($idSector, $ids)
    {
        $len = count($ids);

        $sector = Sector::where('idSetor', $idSector)->first();

        if ($len > 2100) {
            $ids_chunk = array_chunk($ids, 1000);
            foreach ($ids_chunk as $partition_id) {
                Product::whereIn('idProduto', $partition_id)->update(['idSetor' => $idSector, 'dtUltimaAlteracao' => Carbon::now()->toDateTimeString()]);
                FichaTecnica::whereIn('idProduto', $partition_id)->update(['idSetor' => $idSector]);
            }
        } else {
            Product::whereIn('idProduto', $ids)->update(['idSetor' => $idSector, 'dtUltimaAlteracao' => Carbon::now()->toDateTimeString()]);
            FichaTecnica::whereIn('idProduto', $ids)->update(['idSetor' => $idSector]);
        }

        return $sector;
    }

    public function count()
    {
        if (Cache::has('produtos_count')) {
            return Cache::get('produtos_count');
        } else {
            $count = $this->repository->findByField('flAtivo', true)->count();
            Cache::put('produtos_count', $count);
            return $count;
        }

    }

    public function ajustePreco($request)
    {
        $order = !$request['order'] ? 'ASC' : Str::upper($request['order']);

        $res = Product::selectRaw(
            'prd_Produtos.idProduto as id,
            MIN(prd_Codigos.idCodigo) as codigo,
            prd_Produtos.stProdutoAbreviado as descricao,
            convert(decimal(9,2), prd_Produtos.vrUnitario) as preco,
            convert(decimal(9,2), prd_Produtos.vrCusto) as custo,
            prd_Produtos.vrMargemLucro as margem'
        )
            ->join('prd_Codigos', 'prd_produtos.idProduto', '=', 'prd_Codigos.idProduto')
            ->join('prd_Setores', 'prd_Produtos.idSetor', '=', 'prd_Setores.idSetor')
            ->join('prd_Categorias', 'prd_Produtos.idCategoria', '=', 'prd_Categorias.idCategoria')
            ->where('prd_Codigos.idCodigo', '<>', '-100')
            ->where(function ($query) use ($request) {
                if (isset($request['filter']) && strlen($request['filter']) > 0) {
                    $query->where('prd_Codigos.idCodigo', $request['filter'])
                        ->orWhere('prd_Produtos.stProdutoAbreviado', 'LIKE', "%{$request['filter']}%");
                }
            })
            ->where(function ($query) use ($request) {
                if ($request['categoria'] != null && strlen($request['categoria']) > 0)
                    $query->where('prd_Produtos.idCategoria', $request['categoria']);
            })
            ->where('prd_Produtos.flAtivo', true)
            ->orderBy('prd_Produtos.stProdutoAbreviado', $order)
            ->groupBy(['prd_Produtos.idProduto',
                'prd_Produtos.stProdutoAbreviado',
                'prd_Produtos.vrUnitario',
                'prd_Produtos.vrCusto',
                'prd_Produtos.vrMargemLucro'])
            ->paginate($request['limit'], $request['page']);

        return $res;

    }

    public function ajustarPreco($request)
    {
        foreach ($request as $data) {
            $this->repository->update([
                'vrCusto' => $data['custo'],
                'vrUnitario' => $data['preco'],
                'vrMargemLucro' => $data['margem'],
            ], $data['id']);
        }
    }

    private function getIdProdutoByCode($data)
    {
        try {
            return Product::select([
                'prd_Produtos.idProduto',
                'prd_Codigos.idCodigo as codigo',
                'prd_produtos.stProdutoAbreviado as descricao',
                'prd_produtos.nrNcm as ncm',
                'prd_Aliquotas.cfopSaida as cfop',
                'prd_Aliquotas.CstIcms as cstIcms',
                'prd_Aliquotas.AliquotaIcms as aliquotaIcms',
                'prd_Aliquotas.CstPis as cstPis',
                'prd_Aliquotas.AliquotaPis as aliquotaPis',
                'prd_Aliquotas.CstCofins as cstCofins',
                'prd_Aliquotas.AliquotaCofins as aliquotaCofins',
                'prd_Medidas.stMedida as medida',
                'tbt_CargaTributariaMedia.vrAliqNac as ibpt',
                'prd_Produtos.idTipoProduto',
            ])
                ->join('prd_codigos', 'prd_produtos.idProduto', '=', 'prd_Codigos.idProduto')
                ->join('prd_Aliquotas', 'prd_produtos.idAliquota', '=', 'prd_Aliquotas.idAliquota')
                ->join('prd_Medidas', 'prd_produtos.idMedida', '=', 'prd_Medidas.idMedida')
                ->join('tbt_CargaTributariaMedia', 'prd_produtos.nrNcm', '=', 'tbt_CargaTributariaMedia.stCodigo')
                ->where(function ($query) use ($data) {
                    $query->where('prd_Codigos.idCodigo', '=', "$data");
                })->first();
        } catch (Exception $e) {
            throw  new Exception("NATI: Codigo do produto não encontrado");
        }

    }

    public function find($valueSearch)
    {
        if (strlen($valueSearch) > 0) {
            $valueSearch = '%' . $valueSearch . '%';
            $products = Product::join('prd_codigos', 'prd_produtos.idproduto', '=', 'prd_Codigos.idProduto')
                                ->where([['stProduto', 'like', $valueSearch], ['prd_codigos.idCodigo', '<>', '-100']])
                                ->orWhere([['prd_codigos.idCodigo', 'like', $valueSearch], ['prd_codigos.idCodigo', '<>', '-100']])
                                ->get();
        }

        if (count($products)) {
            return json_encode($products);

        } else {
            return null;
        }
    }

    public function findByCod($code)
    {

        if (strlen($code) > 0) {
            $product = Product::select(
                                        'prd_codigos.idCodigo as cod',
                                        'prd_produtos.stProduto as stProduto')
                                ->join('prd_codigos', 'prd_produtos.idProduto', '=', 'prd_Codigos.idProduto')
                                ->where('prd_codigos.idCodigo', $code)
                                ->get();
        }
        return $product;
    }

    public function getProductsByCategory($idCategory)
    {
        try {
            return Product::select([
                'prd_Produtos.idProduto',
                'prd_Codigos.idCodigo as codigo',
                'prd_produtos.stProdutoAbreviado as descricao',
                'prd_produtos.flControlarEstoque as flControlarEstoque'
            ])
                ->join('prd_codigos', 'prd_produtos.idProduto', '=', 'prd_Codigos.idProduto')
                ->where('idCategoria', '=', $idCategory)
                ->where('flAtivo', '=', 1)
                ->orderBy('prd_Produtos.idProduto', 'asc')
                ->get();
        } catch (Exception $e) {
            throw  new Exception("Nenhum produto foi encontrado");
        }
    }

    public function setStockMultiplesProducts (array $products) {
        try {
            foreach($products as $item) {
                $productUpdated = Product::find($item['idProduto']);
                $productUpdated->flControlarEstoque = $item['flControlarEstoque'];
                $productUpdated->save();

                if($item['flControlarEstoque']){
                    $this->stockBalanceService->createStockBalance($item['idProduto']);
                } else {
                    $this->stockBalanceService->deleteStockBalance($item['idProduto']);
                }
            }
        } catch (Exception $e) {
        throw $e;
        }
    }

    public function getProductsWithBalanceStockByCategory($idCategory)
    {
        try {
            return Product::select([
                'prd_Produtos.idProduto',
                'prd_Codigos.idCodigo as codigo',
                'prd_produtos.stProdutoAbreviado as descricao',
                'prd_produtos.flControlarEstoque as flControlarEstoque',
                'est_saldos.idSaldo as idSaldo',
                'est_saldos.nrMinimo as estoqueMinimo'
            ])
                ->join('prd_codigos', 'prd_produtos.idProduto', '=', 'prd_Codigos.idProduto')
                ->join('est_saldos', 'est_saldos.idProduto', '=', 'prd_produtos.idProduto')
                ->where('prd_produtos.idCategoria', '=', $idCategory)
                ->where('prd_produtos.flAtivo', '=', 1)
                ->orderBy('prd_Produtos.idProduto', 'asc')
                ->get();
        } catch (Exception $e) {
            throw  new Exception("Nenhum produto foi encontrado");
        }
    }

    public function sendToMobyo($code)
    {
//        try {
//            return Product::select([
//
//            ]
//                ->join('prd_codigos', 'prd_produtos.idProduto', '=', 'prd_Codigos.idProduto')
//                ->join('prd_categorias', 'prd_produtos.idCategoria', '=', 'prd_categorias.idCategoria')
//                ->
//        }
    }
    public function canCreateProducts() {
        $param = DB::select("select * from sis_parametros where id = 5");
        if(empty($param)) return true;
        if($param[0]->descricao != "Ativar Integraçao Emporio PRO") return true;
        if($param[0]->valor == 'NÃO' || $param[0]->valor != 'SIM') return true;
        return false;
    }
}
