<?php

namespace App\Services\Nati;

use App\Entities\Nati\Stock;
use App\Entities\Nati\StockBalance;
use App\Entities\Nati\StockMovimentation;
use App\Entities\Nati\StockItemsMovimentation;
use App\Repositories\Nati\StockRepository;
use App\Validators\Nati\StockValidator;
use Illuminate\Support\Facades\DB;
use Prettus\Validator\Contracts\ValidatorInterface;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

class StockService
{
    private $stockValidator;
    private $stockBalance;

    public function __construct(StockValidator $stockValidator, StockBalanceService $stockBalance)
    {
        $this->stockValidator = $stockValidator;
        $this->stockBalance = $stockBalance;
    }

    public function all()
    {
        return Stock::all();
    }

    public function getById($id)
    {
        return Stock::with('saldo')->find($id);
    }

    public function create($request)
    {
        return DB::transaction(function () use ($request) {
            // Validação
            $this->stockValidator->with($request)->passesOrFail(ValidatorInterface::RULE_CREATE);

            $request['idEstoque'] = $this->getNextId(); // Gerar próximo ID
            $stock = Stock::create($request);
    
            // $this->stockBalance->setBalance($request['item'], $request['date']); // confirmar oq era isso
    
            return $stock;
        });
    }

    public function update($request, $id)
    {
        return DB::transaction(function () use ($request, $id) {
            // Validação
            $this->stockValidator->with($request)->passesOrFail(ValidatorInterface::RULE_UPDATE);

            // Atualização
            $stock = Stock::findOrFail($id);
            $stock->update($request);

            return $stock;
        });
    }

    public function delete($id)
    {
        return DB::transaction(function () use ($id) {
            $stock = Stock::findOrFail($id);

            // Atualiza o status de ativo
            $stock->update(['flAtivo' => false]);

            return $stock;
        });
    }

    private function getNextId()
    {
        return Stock::max('idEstoque') + 1;
    }

    public function transferStock($idOrigem, $idDestino, array $produtos, $observacao = '')
    {
        DB::beginTransaction();

        try {
            // Verificar se os estoques de origem e destino existem
            $estoqueOrigem = Stock::findOrFail($idOrigem);
            $estoqueDestino = Stock::findOrFail($idDestino);
            $user = Auth::user();

            // Criar a movimentação de estoque
            $movimentacao = StockMovimentation::create([
                'idMovimentacao' => StockMovimentation::max('idMovimentacao') + 1, // Gerar ID único para a movimentação
                'idAbertura' => Carbon::now()->timestamp,
                'idOrigem' => $idOrigem,
                'idDestino' => $idDestino,
                'dtMovimentacao' => Carbon::now(),
                'stOperador' => $user->stApelido,  // Usuário que está realizando a movimentação
                'stNotaFiscal' => '',  // Adicionar se necessário
                'flEntrada' => false,  // Se false, é uma saída do estoque de origem
                'flTipoOrigem' => true, // Tipo de estoque de origem
                'flTipoDestino' => true, // Tipo de estoque de destino
                'flAtivo' => true, // A movimentação está ativa
                'vrEncargos' => 0,  // Encargos, se houver
                'stObservacao' => $observacao,
                'dtData' => Carbon::now(),
                'flTipoEstoque' => true, // Tipo do estoque
                'dtAlteracao' => Carbon::now(),
            ]);

            $nrItem = 1;
            foreach ($produtos as $produto) {
                $saldoOrigem = StockBalance::where('idEstoque', $idOrigem)
                    ->where('idProduto', $produto['idProduto'])
                    ->first();


                if (!$saldoOrigem || $saldoOrigem->nrQuantidade < $produto['nrQuantidade']) {
                    throw new \Exception("Saldo insuficiente para o produto {$produto}");
                }
    
                $novoSaldo = (float) $saldoOrigem->nrQuantidade - (float) $produto['nrQuantidade'];
                DB::table('est_Saldos')
                    ->where('idEstoque', $idOrigem)
                    ->where('idProduto', $produto['idProduto'])
                    ->update(['nrQuantidade' => $novoSaldo]);

                $saldoDestino = StockBalance::where('idEstoque', $idDestino)
                    ->where('idProduto', $produto['idProduto'])
                    ->first();

                if (!$saldoDestino) {
                    $saldoDestino = StockBalance::create([
                        'idSaldo' => StockBalance::max('idSaldo') + 1,
                        'idEstoque' => $idDestino,
                        'idProduto' => $produto['idProduto'],
                        'nrQuantidade' => 0,
                        'nrReservas' => 0,
                        'dtModificacao' => Carbon::now(),
                        'dtConferencia' => Carbon::now(),
                        'nrMinimo' => 0,
                        'dtAlteracao' => Carbon::now(),
                    ]);
                }

                $novoSaldo = (float) $saldoDestino->nrQuantidade + (float) $produto['nrQuantidade'];
                DB::table('est_Saldos')
                    ->where('idEstoque', $idDestino)
                    ->where('idProduto', $produto['idProduto'])
                    ->update(['nrQuantidade' => $novoSaldo]);

                StockItemsMovimentation::create([
                    'idItem' => $nrItem,
                    'idMovimentacao' => $movimentacao->idMovimentacao,
                    'idAbertura' => $movimentacao->idAbertura,
                    'idProduto' => $produto['idProduto'],
                    'nrQuantidade' => $produto['nrQuantidade'],
                    'vrValor' => 0,
                    'nrQuantidadeConversao' => $produto['nrQuantidade'],
                    'stMedidaConversao' => '',
                    'dtAlteracao' => Carbon::now(),
                ]);
                $nrItem++;
                //NAO PRECISA LANCAR UM ITEM MOVIMENTACAO ADICIONANDO NO NOVO ESTOQUE E OUTRA MOVIMENTACAO REMOVENDO DO ESTOQUE DE ORIGEM ?
            }

            DB::commit();
            return true;

        } catch (\Exception $e) {
            DB::rollBack();
            throw new \Exception("Erro ao transferir estoque: " . $e->getMessage());
        }
    }
    public function filterProducts($request)
    {
        $stock = $request['estoque'];
        $filter = $request['filter'];
        $limit = 10;
        $page = 1;

        if(!empty($request->limit)) {
            $limit = $request->limit;
        }
        if(!empty($request->page)) {
            $page = $request->page;
        }

        $products = StockBalance::selectRaw(
            'est_estoques.stEstoque,
            prd_produtos.stProduto,
            est_saldos.nrQuantidade,
            est_saldos.idProduto,
            est_saldos.idEstoque,
            prd_medidas.flFracionado,
            prd_medidas.stMedida,
            prd_medidas.idMedida,
            MAX(prd_codigos.idCodigo) as idCodigo'
        )
            ->join('prd_produtos', 'est_saldos.idProduto', 'prd_produtos.idProduto')
            ->join('prd_codigos', 'prd_produtos.idProduto', 'prd_codigos.idProduto')
            ->join('prd_categorias', 'prd_produtos.idCategoria', 'prd_categorias.idCategoria')
            ->join('prd_medidas', 'prd_produtos.idMedida', 'prd_medidas.idMedida')
            ->join('est_estoques', 'est_saldos.idEstoque', 'est_estoques.idEstoque')
            ->where('est_estoques.idEstoque', $stock)
            ->where(function ($query) use ($filter) {
                    $query->where('prd_Codigos.idCodigo', $filter)
                        ->orWhere('prd_Produtos.stProdutoAbreviado', 'LIKE', "%{$filter}%");
            })
            ->groupBy([
                'est_estoques.stEstoque',
                'prd_produtos.stProduto',
                'est_saldos.nrQuantidade',
                'est_saldos.idProduto',
                'est_saldos.idEstoque',
                'prd_medidas.flFracionado',
                'prd_medidas.stMedida',
                'prd_medidas.idMedida',
                'prd_codigos.idCodigo'
            ])
            ->orderBy('prd_codigos.idCodigo')
            ->paginate($limit, $page);

        return $products;
    }
    protected function getProximoIdMovimentacao()
    {
        return DB::select('SELECT NEXT VALUE FOR sequence_name AS id')[0]->id;
    }
//    private $orderRepository;
//    private $orderValidator;
//    private $stockItemsService;
//    private $stockBalance;
//    private $stockRepository;
//    private $stockValidator;
//
//    public function __construct(StockValidator $stockValidator,StockRepository $stockRepository,StockOrderRepository $orderRepository, StockOrderValidator $orderValidator,
//                                StockItemsMovimentationService $stockItemsService, StockBalanceService $stockBalance)
//    {
//        $this->orderValidator = $orderValidator;
//        $this->orderRepository = $orderRepository;
//        $this->stockItemsService = $stockItemsService;
//        $this->stockBalance = $stockBalance;
//        $this->stockRepository = $stockRepository;
//        $this->stockValidator = $stockValidator;
//        $this->orderRepository->pushCriteria(app('Prettus\Repository\Criteria\RequestCriteria'));
//    }
//
//    public function all()
//    {
//        return $this->stockRepository->all();
////        return $this->repository->with('itens')->all();
//    }
//
//    public function getById($id)
//    {
//        return $this->repository->with('itens')->find($id);
//    }
//
//    public function insertByFechamento($id)
//    {
//        $itens = DB::table('cx_itens')
//            ->join('prd_codigos', 'cx_itens.idProduto', '=', 'prd_codigos.idProduto')
//            ->join('prd_Produtos', 'cx_itens.idProduto', '=', 'prd_Produtos.idProduto')
//            ->join('prd_Medidas', 'prd_Produtos.idMedida', '=', 'prd_Medidas.idMedida')
//            ->select('cx_itens.idProduto as product_id',
//                'prd_codigos.idCodigo as product_code',
//                'prd_Produtos.stProdutoAbreviado as product_name',
//                'prd_Medidas.stMedida as product_uom',
//                'cx_itens.vrCusto as product_cost',
//                'cx_itens.vrUnitario as product_price',
//                DB::raw('SUM(cx_itens.nrQuantidade) as product_quantity'),
//                DB::raw('SUM(cx_itens.vrTotal) as total')
//            )
//            ->where([
//                ['idAbertura', '=', $id],
//                ['cx_itens.flAtivo', '=', true],
//                ['prd_codigos.idCodigo', '<>', '-100']
//            ])
//            ->groupBy('cx_itens.idProduto',
//                'prd_codigos.idCodigo',
//                'prd_Produtos.stProdutoAbreviado',
//                'prd_Medidas.stMedida',
//                'cx_itens.vrUnitario',
//                'cx_itens.vrCusto',
//                'prd_Medidas.stMedida'
//            )
//            ->get();
//
//        $order = new StockOrder();
//        $order->origin = 'PDV';
//        $order->number = StockOrder::max('number') + 1;
//        $order->document = "PDV-" . $id;
//        $order->location_id = 1;
//        $order->user_id = 99;
////       $order->user_id = auth()->user()->idOperador;
//        $order->save();
//
//        $order->addItens($itens);
//
//    }
//
//    public function create($request)
//    {
//        return DB::transaction(function () use ($request) {
//
//            $this->validator->with($request)->passesOrFail(ValidatorInterface::RULE_CREATE);
//
//            $response = $this->repository->create($request);
//
//            //Movimentations Item
//            $this->stockItemsService->save($request['item'], $response['id']);
//
//            //Stock Balance
//            $this->stockBalance->setBalance($request['item'], $request['date']);
//
//            return $response;
//        });
//    }
//
//    public function update($request, $id)
//    {
//        $this->validator->with($request)->passesOrFail(ValidatorInterface::RULE_UPDATE);
//        return $this->repository->update($request, $id);
//    }
//
//    public function delete($id)
//    {
//        return $this->repository->update(['flAtivo' => false], $id);
//    }
//
//    private function getNextId()
//    {
//        return Stock::max('idEstoque') + 1;
//    }
//
//    public function create($request)
//    {
//        $order = new StockOrder();
//        $order->number = StockOrder::max('number') + 1;
//        $order->origin = 1;
//        $order->document = $request->document;
//        $order->location_id = 1;
//        $order->user_id = auth()->user()->idOperador;
//        $order->save();
//
//        if ($request->produtos) {
//            foreach ($request->produtos as $produtos) {
//                $product = new StockOrderIten;
//                $product->order_id = $order->id;
//                $product->product_code = $produtos['product_code'];
//                $product->product_id = $produtos['product_id'];
//                $product->product_name = $produtos['product_name'];
//                $product->product_quantity = $produtos['product_quantity'];
//                $product->product_uom = $produtos['product_uom'];
//                $product->product_price = "95.00";
//                $product->product_cost = $produtos['product_cost'];
//                $product->save();
//
//                DB::table('prd_Produtos')
//                    ->where('idProduto', $produtos['product_id'])
//                    ->update(['vrCusto' => $produtos['product_cost']]);
//            }
//
//            return "Salvei os produtos";
//        }
//
//        return "apenas order";
//
//
//    }
//
//    public function update($data, $id) {
//
//        $this->validator->with($data)->passesOrFail(ValidatorInterface::RULE_UPDATE);
//        return $this->repository->update($data, $id);
//    }
//
//    public function delete($id) {
//
//        $hasSub = SubGrupo::all()->where('idCategoriaConta', $id);
//
//        if (!count($hasSub)) return $this->repository->delete($id);
//        else return false;
//
//    }

}
