<?php

namespace App\Services\Financeiro;

use App\Entities\Financeiro\Lancamento;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use App\Repositories\LancamentosRepository;
use App\Validators\Financeiro\LancamentosValidator;
use Prettus\Validator\Contracts\ValidatorInterface;

class LancamentosService
{
    private $repository;
    private $validator;
    private $contaBancariaService;
    private $itensContaBancariaService;

    public function __construct(LancamentosRepository $repository, ContaBancariaService $contaBancariaService,
                                ItensContaBancariaService $itensContaBancariaService, LancamentosValidator $validator)
    {
        $this->repository = $repository;
        $this->validator = $validator;
        $this->contaBancariaService = $contaBancariaService;
        $this->itensContaBancariaService = $itensContaBancariaService;
    }

    public function get()
    {
        return $this->repository->all();
    }

    public function getLancamento($request)
    {
        $de = Carbon::createFromFormat('d/m/Y', $request->de)->toDateString();
        $de = $de . ' 00:00:00';
        $ate = Carbon::createFromFormat('d/m/Y', $request->ate)->toDateString();
        $ate = $ate . ' 23:59:59';
        $text = '%' . $request->search . '%';

        if ($request->search) {

            if ($request->per == 'all') {
                $lancamento = Lancamento::where('nome_cliente', 'LIKE', $text)
                    ->orWhere('conta', 'LIKE', $text)
                    ->orWhere('sub_grupo', 'LIKE', $text)
                    ->orWhere('grupo', 'LIKE', $text)
                    ->orWhere('stDocumento', 'LIKE', $text)
                    ->orWhere('idCliente', 'LIKE', $text);
            } else {
                if ($request->per == 'idCliente') {
                    $lancamento = Lancamento::where('idCliente', '=', $request->search);
                } else {
                    $lancamento = Lancamento::where($request->per, 'LIKE', $text);
                }
            }

        } else {
            $lancamento = Lancamento::query();
        }

        if ($request->chTipo == 'E') {

            $lancamento->where('dtLancamento', '>=', $de)->where('dtLancamento', '<=', $ate)
                ->orderBy('dtLancamento', 'desc');

        } else if ($request->chTipo == 'V') {

            $lancamento->where('dtVencimento', '>=', $de)->where('dtVencimento', '<=', $ate)
                ->orderBy('dtVencimento', 'desc');

        } else if ($request->chTipo == 'P') {

            $lancamento->where('dtLancamento', '>=', $de)->where('dtLancamento', '<=', $ate)
                ->where('flRecebido', '=', true)
                ->orderBy('dtLancamento', 'desc');
        }

        if ($request->chState == 'T') {
            return $lancamento->get();

        } else if ($request->chState == 'P') {

            return $lancamento->where('flRecebido', '=', true)->get();

        } else if ($request->chState == 'N') {

            return $lancamento->where('flRecebido', '=', false)->get();
        }
    }

    public function setBaixa($request, $idContasReceber, $idCliente)
    {
        $dtRecebimento = Carbon::createFromFormat('d/m/Y', $request['dtRecebimento'])->toDateString();
        $lancamento = Lancamento::where('idContasReceber', '=', $idContasReceber)->where('idCliente', '=', $idCliente);
        $vrDocumento = $lancamento->first()->vrDocumento;
        $vrDiferenca = str_replace(',', '.', $request['vrDiferenca']);

        if ($request['stFormato'] == "M") {
            if ($request['stTipoDiferenca'] == "A") {
                $flJuros = 1;
                $vrDocumento = $vrDocumento + $vrDiferenca;
            } else {
                $flJuros = 0;
                $vrDocumento = $vrDocumento - $vrDiferenca;
            }
        } else {
            if ($request['stTipoDiferenca'] == "A") {
                $flJuros = 1;
                $vrDocumento = $vrDocumento + (($vrDiferenca * $vrDocumento) / 100);
            } else {
                $flJuros = 0;
                $vrDocumento = $vrDocumento - (($vrDiferenca * $vrDocumento) / 100);
            }
        }

        return DB::transaction(function () use ($lancamento, $dtRecebimento, $vrDocumento, $flJuros, $request) {

            if ($request['isConta'] == false) {
                $idContaBancaria = 0;
            } else {
                $idContaBancaria = $request['idContaBancaria'];
                $lanc = $lancamento->first();

                //Atualiza o saldo da conta bancaria
                $this->contaBancariaService->updateSaldoAdd($idContaBancaria, $vrDocumento);

                $itensConta = [
                    'idContaBancaria' => $idContaBancaria,
                    'dtData' => Carbon::now(),
                    'stDescricao' => "A RECEBER - " . $request['sub_grupo'] . " [{$request['nome_cliente']}]",
                    'stDocumento' => $lanc->stDocumento,
                    'vrDocumento' => $vrDocumento,
                    'vrSaldoAnterior' => $request['vrSaldoAnterior'],
                    'idContaPagarReceber' => $lanc->idContasReceber,
                    'idCliente' => $lanc->idCliente,
                    'flCheque' => 0,
                    'dtAlteracao' => Carbon::now(),
                ];
                //itens Conta Bancaria
                $this->itensContaBancariaService->create($itensConta);
            }
            $lancamento->update(['flAtivo' => false,
                'dtRecebimento' => $dtRecebimento,
                'vrRecebido' => $vrDocumento,
                'flRecebido' => 1,
                'flJuros' => $flJuros,
                'stObservacaoBaixa' => isset($request['stObservacaoBaixa']) ? $request['stObservacaoBaixa'] : '',
                'idContaBancaria' => $idContaBancaria]);
        });
    }

    public function selectBaixa($lancamentos)
    {
        $cont = 0;

        foreach ($lancamentos as $lancamento) {

            Lancamento::where('idContasReceber', '=', $lancamento['idContasReceber'])
                ->where('idCliente', '=', $lancamento['idCliente'])->update(['flAtivo' => false,
                    'dtRecebimento' => Carbon::now(),
                    'vrRecebido' => $lancamento['vrDocumento'],
                    'flRecebido' => 1]);

            $cont++;
        }
        return $cont;
    }

    public function getById($idContasReceber, $idCliente)
    {
        $contasReceber = $this->repository->findWhere([
            'idContasReceber' => $idContasReceber,
            'idCliente' => $idCliente
        ])->first();

        return $contasReceber;
    }

    public function insert($request)
    {
        $contasReceber = $this->data($request);

        $this->validator->with($contasReceber)->passesOrFail(ValidatorInterface::RULE_CREATE);

        return $this->repository->create($contasReceber);
    }


    public function atualizar($request, $idContasReceber, $idCliente)
    {
        $contasReceber = $this->data($request);

        $this->validator->with($contasReceber)->passesOrFail(ValidatorInterface::RULE_UPDATE);

        return Lancamento::where('idContasReceber', '=', $idContasReceber)->where('idCliente', '=', $idCliente)->update($contasReceber);

    }

    public function find($text)
    {
        if (strlen($text) > 0) {
            $text = '%' . $text . '%';
            $lancamento = Lancamento::where('stDocumento', 'like', $text)
                ->orWhere('conta', 'like', $text)
                ->orWhere('sub_grupo', 'like', $text)
                ->orWhere('grupo', 'like', $text)
                ->orWhere('nome_cliente', 'like', $text)->get();

            return $lancamento;
        }
    }


    public function delete($idContasReceber, $idCliente)
    {
        return $this->repository->findWhere([
            'idContasReceber' => $idContasReceber,
            'idCliente' => $idCliente
        ])->first()->update(['flAtivo' => false]);

    }

    public function getNextId($cliente)
    {
        if ($this->repository->all()->where('idCliente', $cliente)->last()) {
            return $idContasReceber = $this->repository->all()->where('idCliente', $cliente)->last()->idContasReceber;
        } else {
            return $idContasReceber = 0;
        }
    }

    public function estorno($idContasReceber, $idCliente)
    {

        return DB::transaction(function () use ($idContasReceber, $idCliente) {

            $lancamento = $this->getById($idContasReceber, $idCliente);

            if ($lancamento->idContaBancaria > 0) {
                $this->contaBancariaService->updateSaldoDeduct($lancamento->idContaBancaria, $lancamento->vrRecebido);

                $itensConta = [
                    'idContaBancaria' => $lancamento->idContaBancaria,
                    'dtData' => Carbon::now(),
                    'stDescricao' => "A RECEBER - " . $lancamento->sub_grupo,
                    'stDocumento' => "ESTORNO - {$lancamento->stDocumento}",
                    'vrDocumento' => $lancamento->vrRecebido * (-1),
                    'vrSaldoAnterior' => 0,
                    'idContaPagarReceber' => $lancamento->idContasReceber,
                    'idCliente' => $lancamento->idCliente,
                    'flCheque' => 0,
                    'dtAlteracao' => Carbon::now(),
                ];

                //itens Conta Bancaria
                $this->itensContaBancariaService->create($itensConta);
            }
            Lancamento::where('idContasReceber', '=', $idContasReceber)->where('idCliente', '=', $idCliente)
                ->update(['flAtivo' => true, 'dtRecebimento' => null, 'flRecebido' => 0, 'vrRecebido' => 0.00]);
        });
    }

    public function data($request)
    {

        $dtLancamento = Carbon::createFromFormat('d/m/Y', $request['dtLancamento'])->toDateString();
        $dtVencimento = Carbon::createFromFormat('d/m/Y', $request['dtVencimento'])->toDateString();

        $contasReceber = [
            "idContasReceber" => isset($request['idContasReceber']) ? $request['idContasReceber'] : $this->getNextId($request['idCliente']) + 1,
            "idCliente" => $request['idCliente'],
            "idTipoConta" => $request['idSub_grupo'],
            "dtLancamento" => $dtLancamento,
            "dtVencimento" => $dtVencimento,
            "dtRecebimento" => isset($request['dtRecebimento']) ? $request['dtRecebimento'] : null,
            "nrParcela" => 1,
            "nrTotalParcelas" => $request['nrTotalParcelas'],
            "vrDocumento" => str_replace(',', '.', $request['vrFinal']),
            "vrRecebido" => $request['vrRecebido'],
            "vrJuros" => 0.00,
            "flJuros" => null,
            "flRecebido" => $request['flRecebido'],
            "flAtivo" => $request['flAtivo'],
            "stOperador" => $request['stOperador'],
            "stObservacao" => strtoupper($request['stObservacao']),
            "stDocumento" => $request['stDocumento'],
            "stTipoBaixa" => 'I',
            "stTipoJuros" => 'P',
            "idFormaPagamento" => isset($request['idFormaPagamento']) ? $request['idFormaPagamento'] : 1,
            "stObservacaoBaixa" => isset($request['stObservacaoBaixa']) ? $request['stObservacaoBaixa'] : null,
            "vrDescontoLancamento" => str_replace(',', '.', $request['vrDescontoLancamento']),
            "idContaBancaria" => 0,
            "idConta" => $request['idConta'],
            "dtAlteracao" => Carbon::now(),
            "idTipopagamento" => 1,
            "stFormato" => $request['stFormato']
        ];

        return $contasReceber;
    }

}
