<?php

namespace App\Http\Controllers;

use Exception;
use App\Entities\Logs;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Routing\Controller;
use App\Services\Nati\ProductService;
use App\Http\Requests\Nati\ProductCreateRequest;
use App\Http\Requests\Nati\ProductUpdateRequest;
use Prettus\Validator\Exceptions\ValidatorException;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Symfony\Component\HttpFoundation\Exception\BadRequestException;

class ProductsController extends Controller
{
    /**
     * @var ProductService
     */
    protected $service;

    public function __construct(ProductService $service)
    {
        $this->service = $service;
    }

    /**
     * Display a listing of the resource.
     * @param Request $request
     * @return JsonResponse
     */
    public function index(Request $request)
    {
        try {
            $response = $this->service->get($request->query());
            return response()->json($response);

        } catch (Exception $exception) {
            return response()->json([
                'error' => true,
                'message' => $exception->getMessage()
            ], 404);
        }
    }

    public function download()
    {
        $file = public_path() . "/download/modelo_importacao.csv";

        $headers = array(
            'Content-Type: application/csv',
        );

        return response()->download($file, 'modelo_importacao.csv', $headers);
    }

    public function updateTaxData(Request $request, $id)
    {
        try {
            return response()->json($this->service->updateTaxData($id, $request->all()));
        } catch (Exception $exception) {
            return response()->json([
                'error' => true,
                'message' => $exception->getMessage()
            ], 404);
        }
    }

    public function updateTaxDatas(Request $request)
    {
        try {
            return response()->json($this->service->updateTaxDatas($request->all()));
        } catch (Exception $exception) {
            return response()->json([
                'error' => true,
                'message' => $exception->getMessage()
            ], 404);
        }
    }

    public function painelAliquotas(Request $request)
    {
        try {
            return response()->json($this->service->getPainelAliquotas($request->all()));
        } catch (Exception $exception) {
            return response()->json([
                'error' => true,
                'message' => $exception->getMessage()
            ], 404);
        }
    }

    public function verifyCode($id)
    {
        return $this->service->verifyCode($id);
    }

    public function nextcode()
    {
        return $this->service->nextcode();
    }

    public function export()
    {
        try {
            return $this->service->export();
        } catch (Exception $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessageBag()
            ], 400);
        }
    }

    public function exportForIntegration()
    {
        try {
            return $this->service->exportForIntegration();
        } catch (Exception $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessageBag()
            ], 400);
        }

    }

    public function exportForMobyo()
    {
        try {
            return $this->service->exportForMobyo();
        } catch (Exception $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessage()
            ], 400);
        }
    }
    public function exportForLinx()
    {
        try {
            return $this->service->exportForLinx();
        } catch (Exception $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessage()
            ], 400);
        }
    }

    public function import(Request $request)
    {
        try {
            $response = $this->service->import($request);

            $user = auth()->user();
            Logs::writeLog($user->stApelido, "PRODUTOS", 'O USUARIO [' . $user->stApelido . '] IMPORTOU UMA PLANILHA DE PRODUTOS');

            return response()->json(['sucesso' => true, 'data' => $response], 201);
        } catch (Exception $exception) {
            $message = $exception->getMessage();
            return response()->json([
                'error' => true,
                'message' => $message,
            ], 500);
        }

    }

//    public function import(Request $request)
//    {
//        try {
//            $path = $request->file('file')->getRealPath();
//            $data = $this->csvToArray($path);
//
//            $count = 0;
//            foreach ($data as $row) {
//                $this->service->import($row);
//                $count++;
//            }
//            $user = auth()->user();
//            Logs::writeLog($user->stApelido, "PRODUTOS", 'O USUARIO [' . $user->stApelido . '] IMPORTOU O ARQUIVO [' . $path . ']');
//
//            return response()->json(['sucesso' => true, 'produtos' => $count], 201);
//        } catch (Exception $exception) {
//            return response()->json([
//                'error' => true,
//                'message' => $exception->getMessage()
//            ], 500);
//        }
//
//    }

//    function csvToArray($filename = '', $delimiter = ';')
//    {
//        if (!file_exists($filename) || !is_readable($filename))
//            return false;
//
//        $header = null;
//        $data = array();
//        if (($handle = fopen($filename, 'r')) !== false) {
//            while (($row = fgetcsv($handle, 1000, $delimiter)) !== false) {
//                if (!$header)
//                    $header = $row;
//                else {
//                    if (!$row[0]) continue;
//                    $data[] = array_combine($header, $row);
//                }
//            }
//            fclose($handle);
//        }
//
//        return $data;
//    }

    /**
     * Store a newly created resource in storage.
     * @param ProductCreateRequest $request
     * @return JsonResponse
     */
    public function store(ProductCreateRequest $request)
    {
        try {
            $user = auth()->user();
            $response = $this->service->create($request->all());
            Logs::writeLog($user->stApelido, "PRODUTOS", 'O USUARIO [' . $user->stApelido . '] CADASTROU O PRODUTO [' . $response->stProdutoAbreviado . ']');

            return response()->json([
                'data' => $response
            ], 201);

        } catch (ValidatorException $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessage()
            ], 400);
        } catch (Exception $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessage()
            ], 400);
        }
    }

    public function filter(string $data)
    {
        try {
            return response()->json($this->service->filter($data));
        } catch (Exception $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessage()
            ], 404);
        }
    }

    public function ajustePreco(Request $request)
    {
        try {
            $user = auth()->user();
            $response =$this->service->ajustePreco($request->all());

            return response()->json($response);
        } catch (Exception $exception) {
            return response()->json([
                'error' => true,
                'message' => $exception->getMessage()
            ], 404);
        }
    }

    public function ajustarPreco(Request $request)
    {
        try {
            return response()->json($this->service->ajustarPreco($request->all()));
        } catch (Exception $exception) {
            return response()->json([
                'error' => true,
                'message' => $exception->getMessage()
            ], 404);
        }
    }

    /**
     * Show the specified resource.
     * @return JsonResponse
     */
    public function show($id)
    {
        try {
            $response = $this->service->getById($id);
            return response()->json($response);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessage()
            ], 404);

        } catch (ValidatorException $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessageBag()
            ], 400);
        }
    }

    public function count()
    {
        try {
            return response()->json($this->service->count());
        } catch (Exception $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessage()
            ], 404);
        }
    }

    /**
     * Update the specified resource in storage.
     * @param ProductUpdateRequest $request
     * @param                      $id
     * @return JsonResponse
     */
    public function update(ProductUpdateRequest $request, $id)
    {
        try {
            $user = auth()->user();
            $response = $this->service->update($request->all(), $id);
            Logs::writeLog($user->stApelido, "PRODUTOS", 'O USUARIO [' . $user->stApelido . '] ALTEROU O PRODUTO [' . $response->stProdutoAbreviado . ']');

            return response()->json($response);

        } catch (ModelNotFoundException $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessage()
            ], 404);

        } catch (ValidatorException $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessageBag()
            ], 400);
        }
    }

    public function destroy($id)
    {
        try {
            $user = auth()->user();
            $response = $this->service->destroy($id);
            Logs::writeLog($user->stApelido, "PRODUTOS", 'O OPERADOR [' . $user->stApelido . '] DESATIVOU O PRODUTO [ID ' . $response->idProduto . ' - ' . $response->stProdutoAbreviado . ']');

            return response()->json([
                'data' => $response,
                'message' => 'Success'
            ]);

        } catch (ModelNotFoundException $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessage()
            ], 404);
        }
    }

    public function active($id)
    {
        try {
            $user = auth()->user();
            $response = $this->service->active($id);
            Logs::writeLog($user->stApelido, "PRODUTOS", 'O OPERADOR [' . $user->stApelido . '] ATIVOU O PRODUTO [ID ' . $response->idProduto . ' - ' . $response->stProdutoAbreviado . ']');
            return response()->json([
                'data' => $response,
                'message' => 'product active success'
            ]);

        } catch (ModelNotFoundException $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessage()
            ], 404);
        }
    }

    public function activeStock($id)
    {
        try {
            $response = $this->service->activeStock($id);
            return response()->json([
                'message' => 'stock active success'
            ]);

        } catch (ModelNotFoundException $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessage()
            ], 404);
        }
    }

    public function getByGroup(Request $request, $id)
    {
        try {
            $result = $this->service->getByGroup($id, $request->get('group'));

            $map = $result->map(function ($item) {
                return [
                    'id' => $item->idProduto,
                    'code' => $item->code,
                    'description' => $item->stProdutoAbreviado,
                    'value' => $item->vrUnitario
                ];
            });

            return response()->json($map);
        } catch (ModelNotFoundException $notFoundException) {
            return response()->json([
                'error' => true,
                'message' => $notFoundException->getMessage()
            ], 404);
        } catch (BadRequestException $badRequestException) {
            return response()->json([
                'error' => true,
                'message' => $badRequestException->getMessage()
            ], 400);
        }
    }

    public function getPrintConfig(Request $request)
    {
        try {
            return response()->json($this->service->getPrintConfig($request->all()));
        } catch (Exception $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessage()
            ], 404);
        }
    }

    public function moveCategory(Request $request, $category, $sub_category)
    {
        try {
            return response()->json($this->service->moveCategory($category, $sub_category, $request->get('ids')));
        } catch (\Exception $exception) {
            return response()->json([
                'error' => true,
                'message' => $exception->getMessage()
            ], 400);
        }
    }

    public function moveSector(Request $request, $sector)
    {
        try {
            $user = auth()->user();
            $response = $this->service->moveSector($sector, $request->get('ids'));
            Logs::writeLog($user->stApelido, "SETORES", 'O USUARIO [' . $user->stApelido . '] MOVEU OS PRODUTOS DO SETOR [' . $response->stSetor . ']');
        } catch (Exception $exception) {
            return response()->json([
                'error' => true,
                'message' => $exception->getMessage()
            ], 400);
        }
    }

    public function find($valueSearch)
    {
        try {
            return $this->service->find($valueSearch);
        } catch (Exception $exception) {
            return response()->json([
                'error' => true,
                'message' => $exception->getMessage()
            ], 400);
        }
    }

    public function findByCod($code)
    {
        try {
            $result = $this->service->findByCod($code);

            $map = $result->map(function ($item) {
                return [
                    'code' => $item->cod,
                    'stProduto' => $item->stProduto
                ];
            });
            // dd($map);
            return response()->json($map
        );
        } catch (Exception $exception) {
            return response()->json([
                'error' => true,
                'message' => $exception->getMessage()
            ], 400);
        }
    }


    public function getProductsByCategory($idCategory)
    {
        try {
            return $this->service->getProductsByCategory($idCategory);
        } catch (Exception $exception) {
            return response()->json([
                'error' => true,
                'message' => $exception->getMessage()
            ], 400);
        }
    }

    public function setStockMultiplesProducts(Request $request)
    {
        try {
            return $this->service->setStockMultiplesProducts($request->all());
        } catch (Exception $exception) {
            return response()->json([
                'error' => true,
                'message' => $exception->getMessage()
            ], 400);
        }
    }

    public function getProductsWithBalanceStockByCategory($idCategory)
    {
        try {
            return $this->service->getProductsWithBalanceStockByCategory($idCategory);
        } catch (Exception $exception) {
            return response()->json([
                'error' => true,
                'message' => $exception->getMessage()
            ], 400);
        }
    }

    public function sendToMobyo($code)
    {
        try {
            return $this->service->sendToMobyo($code);
        } catch (Exception $exception) {
            return response()->json([
                'error' => true,
                'message' => $exception->getMessage()
            ], 500);
        }
    }

    public function canCreateProducts()
    {
        try {
            return $this->service->canCreateProducts();
        } catch (Exception $exception) {
            return response()->json([
                'error' => true,
                'message' => $exception->getMessage()
            ], 500);
        }
    }
}
