<?php

namespace App\Services\Nati;

use App\Entities\Nati\RespostaProduto;
use App\Repositories\Nati\RespostaProdutoRepository;
use App\Services\Service;
use App\Validators\Nati\RespostaProdutoValidator;
use Carbon\Carbon;
use Illuminate\Support\Str;
use App\Entities\Nati\Categoria;
use Prettus\Validator\Contracts\ValidatorInterface;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;
use App\Services\Nati\ProductService;

class RespostaProdutoService extends Service
{
    private $repository;
    private $validator;

    /**
     * CategoriaService constructor.
     * @param RespostaProdutoRepository $repository
     * @param RespostaProdutoValidator $validator
     */
    public function __construct(RespostaProdutoRepository $repository, RespostaProdutoValidator $validator, ProductService $productService)
    {
        $this->repository = $repository;
        $this->validator = $validator;
        $this->productService = $productService;
    }


    public function getAll()
    {
        $response = $this->repository->with('respostas')
            ->orderBy('stObservacao','asc')
            ->findWhere(['flAtivo' => true]);
        return $response;
    }


    public function store($data)
    {

        $this->validator->with($data)->passesOrFail(ValidatorInterface::RULE_CREATE);
        $arr = [
            'idProduto' => $data['idProduto'],
            'vrUnitario' => number_format($data['vrUnitario'], 2, '.', ''),
            'flAtivo' => 1,
            'dtAlteracao' => Carbon::now(),
        ];
        $respostaProduto = $this->repository->create($arr);
        return $respostaProduto;
    }

    public function update($request, $id)
    {
        $this->validator->with($request)->passesOrFail(ValidatorInterface::RULE_UPDATE);

        $request['stObservacao'] = Str::upper($request['stObservacao']);
        $result = $this->repository->update($request, $id);
        return $result;
    }


    public function delete($id)
    {
        $observacao = $this->repository->find($id);

        if ($observacao->respostas->count() > 0) {
            throw new BadRequestHttpException('Não é possivel desativar uma observação que está vinculada a respostas');
        } else {
            $observacao->flAtivo = false;
            $observacao->save();
            return $observacao;
        }
    }


    public function getOrCreate($categoria)
    {
        $cat = $this->repository->where('stCategoria', strtoupper($categoria))->first();
        if($cat){
            return $cat;
        }else{
            return $this->create(['stCategoria' => strtoupper($categoria)]);
        }
    }

    public function filter($data)
    {
        return RespostaObservacao::where('stObservacao', 'LIKE', "%$data%")
            ->where('flAtivo', true)
            ->paginate(10);
    }
    public function show($id)
    {
        return $this->repository->with('respostas')->find($id);
    }
    public function deleteByIdRespostaProduto($idRespostaProduto)
    {
        $respostas = $this->repository->findByField('idRespostaProduto', $idRespostaProduto);
        foreach ($respostas as $resposta) {
            $this->repository->where('idRespostaProduto', $resposta->idRespostaProduto)->delete();
        }
    }
    public function getWithProductByIdRespostaProduto($idRespostaProduto)
    {
        $respostaProduto = $this->repository->find($idRespostaProduto);
        $product = $this->productService->getById($respostaProduto['idProduto']);
        $respostaProduto['stProduto'] = $product['stProduto'];
        $respostaProduto['codigo'] = $product->codigos[0]['idCodigo'];
        return $respostaProduto;
    }
}
