<?php

namespace App\Entities\Nati\ImportProducts;

use App\Entities\Nati\Product;
use App\Repositories\Nati\ProductRepository;
use App\Services\Nati\AliquotaService;
use App\Services\Nati\CategoriaService;
use App\Services\Nati\MeasureService;
use App\Services\Nati\SectorService;
use App\Services\Nati\SubCategoriaService;
use App\Validators\Nati\ProductValidator;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Prettus\Validator\Contracts\ValidatorInterface;
use App\Entities\Nati\Code;
use Exception;
use Symfony\Component\HttpFoundation\Exception\BadRequestException;

class ImportProducts
{
    protected $repository;
    protected $validator;
    protected $categoriaService;
    protected $aliquotaService;
    protected $setorService;
    protected $medidaService;
    protected $subCategoriaService;
    protected $duplicateProducts = [];

    public function __construct(
        ProductRepository $repository,
        ProductValidator $validator,
        CategoriaService $categoriaService,
        AliquotaService $aliquotaService,
        SectorService $setorService,
        MeasureService $medidaService,
        SubCategoriaService $subCategoriaService
    )
    {
        $this->repository = $repository;
        $this->validator = $validator;
        $this->categoriaService = $categoriaService;
        $this->aliquotaService = $aliquotaService;
        $this->setorService = $setorService;
        $this->medidaService = $medidaService;
        $this->subCategoriaService = $subCategoriaService;
    }

    public function importFile($pathFile)
    {
        $arrayProducts = $this->convertCsvToArray($pathFile);
        return $response = $this->insertProducts($arrayProducts);
    }

    private function convertCsvToArray($filename = '', $delimiter = ';')
    {
        if (!file_exists($filename) || !is_readable($filename))
            return false;
        $header = null;
        $data = array();
        if (($handle = fopen($filename, 'r')) !== false) {
            while (($row = fgetcsv($handle, 1000, $delimiter)) !== false) {
                if (!$header)
                    $header = $row;
                else {
                    if (!$row[0]) continue;
                    $data[] = array_combine($header, $row);
                }
            }
            fclose($handle);
        }
        return $data;
    }

    private function insertProducts($arrayProducts)
    {
        $qtdProductsInserted = 0;
        foreach ($arrayProducts as $item) {
            DB::beginTransaction();
            $product= $this->parseProduct($item);
            $this->validator->with($product)->passesOrFail(ValidatorInterface::RULE_CREATE);
            $product = $this->repository->create($product);
            $this->insertCode($item, $product->idProduto);
            DB::commit();
            $qtdProductsInserted++;
        }
        $res['inseridos'] = $qtdProductsInserted;
        if($this->duplicateProducts) {
            $res['duplicados'] = $this->duplicateProducts;
            $res['inseridos'] = $res['inseridos'] - count($this->duplicateProducts);
        }
        return $res;
    }

    private function insertCategory($data)
    {
        try {
            return $categoria = $this->categoriaService->getOrCreate((empty($data['Categoria'])) ? $data['Categoria'] = 'SEM CATEGORIA' : $this->trataString(utf8_encode($data['Categoria'])));
        } catch (Exception $exception) {
            throw new Exception("Categoria: [" . $data['Categoria'] . "] invalida");
        }
    }
    private function insertSubCategory($data, $categoria)
    {
        try {
            return $subCategoria = $this->subCategoriaService->getOrCreate((empty($data['Subcategoria'])) ? $data['Subcategoria'] = 'SEM SUBCATEGORIA' : $this->trataString(utf8_encode($data['Subcategoria'])), $categoria->idCategoria);
        } catch (Exception $exception) {
            throw new Exception("SubCategoria: [" . $data['Subcategoria'] . "] invalida");
        }
    }
    private function insertSector($data)
    {
        try {
            return $setor = $this->setorService->getOrCreate((empty($data['Setor'])) ? $data['Setor'] = 'SEM SETOR' : $this->trataString(utf8_encode($data['Setor'])));
        } catch (Exception $exception) {
            throw new Exception("Setor: [" . $data['Setor'] . "] invalido");
        }
    }
    private function insertMedida($data)
    {
        try {
            return $this->medidaService->getOrCreate((empty($data['Medida'])) ? $data['Medida'] = 'UN' : $this->trataString(utf8_encode($data['Medida'])));
        } catch (Exception $exception) {
            throw new Exception("Medida: [" . $data['Medida'] . "] invalida");
        }
    }
    private function insertAliquota($data)
    {
        try {
            return $aliquota = $this->aliquotaService->getOrCreate($data);
        } catch (Exception $exception) {
            throw new Exception("O item [" . $data['Descricao'] . "] possui uma aliquota invalida");
        }
    }


    private function parseProduct($data)
    {
        $data['CST-ICMS'] = str_pad(utf8_encode($data['CST-ICMS']), 2, '0', STR_PAD_LEFT);
        $data['CST-PIS'] = str_pad($data['CST-PIS'], 2, '0', STR_PAD_LEFT);
        $data['CST-COFINS'] = str_pad($data['CST-COFINS'], 2, '0', STR_PAD_LEFT);

        $categoria = $this->insertCategory($data);
        $setor = $this->insertSector($data);
        $medida = $this->insertMedida($data);
        $aliquota = $this->insertAliquota($data);
        $subCategoria = $this->insertSubCategory($data, $categoria);

        $product = [];
        $product['idProduto'] = isset($data['idProduto']) ? $data['idProduto'] : $this->getNextId();
        $product['stProduto'] = $this->trataString(utf8_encode(strtoupper(Str::substr($data['Descricao'], 0, 49))));
        $product['stProdutoAbreviado'] = $this->trataString(utf8_encode(strtoupper(Str::substr($data['Descricao'], 0, 27))));
        $product['stOperador'] = auth()->user()->stApelido;
        $product['idCategoria'] = $categoria->idCategoria;
        $product['idSetor'] = $setor->idSetor;
        $product['idMedida'] = $medida->idMedida;
        $product['vrUnitario'] = str_replace(',','.', $data['Preco']);
        $product['vrCusto'] = empty($data['Custo']) ? $data['Custo'] = 0 : str_replace(',', '.', $data['Custo']);
        $product['vrMargemLucro'] = empty($data['Margem']) ? $data['vrMargemLucro'] = 0 : $data['Margem'];
        $product['vrUnitarioPromocao'] = str_replace(',','.', $data['Preco']);
        $product['vrUnitarioPromocaoHora'] = str_replace(',','.', $data['Preco']);
        $product['vrUnitario2'] = str_replace(',','.', $data['Preco']);
        $product['idTipoProduto'] = 1;
        $product['stCest'] = isset($data['CEST']) ? $data['CEST'] : null;
        $product['nrNCM'] = empty($data['NCM']) ? $data['NCM'] = '19059090' : str_replace('.', '', $data['NCM']);
        $product['idAliquota'] = $aliquota->idAliquota;
        $product['flComissao'] = true;
        $product['idLocalizacao'] = 1;
        $product['idSubCategoria'] = $subCategoria['idSubCategoria'];
        $product['dtUltimaAlteracao'] = Carbon::now();
        $product['dtAlteracao'] = Carbon::now();
        $product['dtInicioPromocao'] = '2008-01-01 00:00:00.000';
        $product['dtFimPromocao'] = '2008-01-01 00:00:00.000';
        $product['dtInicioPromocaoHora'] = '2008-01-01 00:00:00.000';
        $product['dtFimPromocaoHora'] = '2008-01-01 00:00:00.000';
        $product['flPocket'] = true;
        $product['nr10Mais'] = 0;
        $product['flPromocao'] = false;
        $product['nrValidade'] = 0;
        $product['flControlarEstoque'] = false;
        $product['flVendaCartao'] = true;
        $product['flVendaTicket'] = false;
        $product['flVendaSmart'] = false;
        $product['flRequererItemAdicional'] = false;
        $product['nrFatorConversao'] = 0;
        $product['nrTipoProduto'] = 0;
        $product['idProdutoVinculado'] = 0;
        $product['flCupomPreVenda'] = false;
        $product['flPromocaoHora'] = false;
        $product['stDescricao'] = '';
        $product['flAtivo'] = true;

        return $product;
    }

    private function insertCode($product, $idproduto)
    {
        Code::where('idProduto', $idproduto)->delete();

            if (!isset($product['Codigo']) && !isset($product['EAN'])) {
                throw new Exception("Nenhum codigo foi informado para o produto");
            }
            if (isset($product['Codigo']) && Str::length($product['Codigo']) > 0) {
                $product['Codigo'] = $this->trataCodigo($product['Codigo']);
                if(!$this->alreadyExists($product['Codigo'])) {
                    Code::create(['idCodigo' => $product['Codigo'], 'idProduto' => $idproduto]);
                } else {
                    $duplicate = [];
                    $duplicate['produto'] = $product['Codigo'] . ' - '. $product['Descricao'];
                    array_push($this->duplicateProducts, $duplicate);
                    DB::rollBack();
                };
            }
            if (isset($product['EAN']) && Str::length($product['EAN'])) {
                $product['EAN'] = $this->trataCodigo($product['EAN']);
                if(!$this->alreadyExists($product['EAN'])) {
                    Code::create(['idCodigo' => $product['EAN'], 'idProduto' => $idproduto]);
                } else {
                    $duplicate = [];
                    $duplicate['produto'] = $product['Codigo'] . ' - '. $product['Descricao'];
                    array_push($this->duplicateProducts, $duplicate);
                    DB::rollBack();
                };
            }
    }

    private function trataCodigo($codigo)
    {
        $codigo = $this->trataString($codigo);
        return $codigo;
    }

    private function trataString($string)
    {
        $comAcentos = array('à', 'á', 'â', 'ã', 'ä', 'å', 'ç', 'è', 'é', 'ê', 'ë', 'ì', 'í', 'î', 'ï', 'ñ', 'ò', 'ó', 'ô', 'õ', 'ö', 'ù', 'ü', 'ú', 'ÿ', 'À', 'Á', 'Â', 'Ã', 'Ä', 'Å', 'Ç', 'È', 'É', 'Ê', 'Ë', 'Ì', 'Í', 'Î', 'Ï', 'Ñ', 'Ò', 'Ó', 'Ô', 'Õ', 'Ö', 'Ù', 'Ü', 'Ú', '-', '/', '.', ',');
        $semAcentos = array('a', 'a', 'a', 'a', 'a', 'a', 'c', 'e', 'e', 'e', 'e', 'i', 'i', 'i', 'i', 'n', 'o', 'o', 'o', 'o', 'o', 'u', 'u', 'u', 'y', 'A', 'A', 'A', 'A', 'A', 'A', 'C', 'E', 'E', 'E', 'E', 'I', 'I', 'I', 'I', 'N', 'O', 'O', 'O', 'O', 'O', 'U', 'U', 'U', '', '', '', '');

        return str_replace($comAcentos, $semAcentos, $string);
    }

    private function getNextId()
    {
        return Product::max('idProduto') + 1;
    }

    public function alreadyExists($code)
    {
       $result = DB::select("select idCodigo from prd_codigos where idCodigo = '$code'");
       if($result) return true;
       return false;
    }
}
