<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;

use App\Http\Requests;
use Prettus\Validator\Contracts\ValidatorInterface;
use Prettus\Validator\Exceptions\ValidatorException;
use App\Http\Requests\DlvPaymentCreateRequest;
use App\Http\Requests\DlvPaymentUpdateRequest;
use App\Repositories\PaymentRepository;
use App\Validators\DlvPaymentValidator;

/**
 * Class DlvPaymentsController.
 *
 * @package namespace App\Http\Controllers;
 */
class DlvPaymentsController extends Controller
{
    /**
     * @var PaymentRepository
     */
    protected $repository;

    /**
     * @var DlvPaymentValidator
     */
    protected $validator;

    /**
     * DlvPaymentsController constructor.
     *
     * @param PaymentRepository   $repository
     * @param DlvPaymentValidator $validator
     */
    public function __construct(PaymentRepository $repository, DlvPaymentValidator $validator)
    {
        $this->repository = $repository;
        $this->validator  = $validator;
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $this->repository->pushCriteria(app('Prettus\Repository\Criteria\RequestCriteria'));
        $dlvPayments = $this->repository->all();

        if (request()->wantsJson()) {

            return response()->json([
                'data' => $dlvPayments,
            ]);
        }

        return view('dlvPayments.index', compact('dlvPayments'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  DlvPaymentCreateRequest $request
     *
     * @return \Illuminate\Http\Response
     *
     * @throws \Prettus\Validator\Exceptions\ValidatorException
     */
    public function store(DlvPaymentCreateRequest $request)
    {
        try {

            $this->validator->with($request->all())->passesOrFail(ValidatorInterface::RULE_CREATE);

            $dlvPayment = $this->repository->create($request->all());

            $response = [
                'message' => 'DlvPayment created.',
                'data'    => $dlvPayment->toArray(),
            ];

            if ($request->wantsJson()) {

                return response()->json($response);
            }

            return redirect()->back()->with('message', $response['message']);
        } catch (ValidatorException $e) {
            if ($request->wantsJson()) {
                return response()->json([
                    'error'   => true,
                    'message' => $e->getMessageBag()
                ]);
            }

            return redirect()->back()->withErrors($e->getMessageBag())->withInput();
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int $id
     *
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $dlvPayment = $this->repository->find($id);

        if (request()->wantsJson()) {

            return response()->json([
                'data' => $dlvPayment,
            ]);
        }

        return view('dlvPayments.show', compact('dlvPayment'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int $id
     *
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $dlvPayment = $this->repository->find($id);

        return view('dlvPayments.edit', compact('dlvPayment'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  DlvPaymentUpdateRequest $request
     * @param  string            $id
     *
     * @return Response
     *
     * @throws \Prettus\Validator\Exceptions\ValidatorException
     */
    public function update(DlvPaymentUpdateRequest $request, $id)
    {
        try {

            $this->validator->with($request->all())->passesOrFail(ValidatorInterface::RULE_UPDATE);

            $dlvPayment = $this->repository->update($request->all(), $id);

            $response = [
                'message' => 'DlvPayment updated.',
                'data'    => $dlvPayment->toArray(),
            ];

            if ($request->wantsJson()) {

                return response()->json($response);
            }

            return redirect()->back()->with('message', $response['message']);
        } catch (ValidatorException $e) {

            if ($request->wantsJson()) {

                return response()->json([
                    'error'   => true,
                    'message' => $e->getMessageBag()
                ]);
            }

            return redirect()->back()->withErrors($e->getMessageBag())->withInput();
        }
    }


    /**
     * Remove the specified resource from storage.
     *
     * @param  int $id
     *
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $deleted = $this->repository->delete($id);

        if (request()->wantsJson()) {

            return response()->json([
                'message' => 'DlvPayment deleted.',
                'deleted' => $deleted,
            ]);
        }

        return redirect()->back()->with('message', 'DlvPayment deleted.');
    }
}
