<?php

namespace App\Http\Controllers;

use App\Http\Requests\ProfileCreateRequest;
use App\Http\Requests\ProfileUpdateRequest;
use App\Entities\Profile;
use App\Services\ProfileService;
use App\Entities\Logs;
use Illuminate\Http\Request;
use Prettus\Validator\Contracts\ValidatorInterface;
use Prettus\Validator\Exceptions\ValidatorException;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

/**
 * Class ProfilesController.
 *
 * @package namespace App\Http\Controllers;
 */
class ProfilesController extends Controller
{
    protected $service;

    public function __construct(ProfileService $service)
    {
        $this->service = $service;
    }

    public function index(Request $request)
    {
        try {
            return response()->json($this->service->get($request));

        } catch (\Exception $exception) {
            return response()->json([
                'error' => true,
                'message' => $exception->getMessage()
            ], 404);
        }
    }

    public function permissions()
    {
        try {
            return response()->json($this->service->getPermissions());

        } catch (\Exception $exception) {
            return response()->json([
                'error' => true,
                'message' => $exception->getMessage()
            ], 404);
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  ProfileCreateRequest $request
     *
     * @return \Illuminate\Http\JsonResponse
     *
     * @throws \Prettus\Validator\Exceptions\ValidatorException
     */
    public function store(ProfileCreateRequest $request)
    {
        try {
            $user = auth()->user();
            $response = $this->service->insert($request->all());
            Logs::writeLog($user->stApelido, "PERFIS", 'O OPERADOR [' . $user->idOperador . '-' . $user->stApelido . '] CADASTROU O PERFIL [' . $response->idPerfil . '-' . $response->stDescricao . ']');

        } catch (\Exception $exception) {
            return response()->json([
                'error' => true,
                'message' => $exception->getMessage()
            ], 404);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int $id
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function show($id)
    {
        try {
            return response()->json($this->service->getById($id));

        } catch (\Exception $exception) {
            return response()->json([
                'error' => true,
                'message' => $exception->getMessage()
            ], 404);
        }
    }

    public function update(ProfileUpdateRequest $request, $id)
    {
        try {
            $user = auth()->user();
            $response = $this->service->update($request->all(), $id);
            Logs::writeLog($user->stApelido, "PERFIS", 'O OPERADOR [' . $user->idOperador . '-' . $user->stApelido . '] EDITOU O PERFIL [' . $id . ' - ' . $request->stDescricao . ']');

        } catch (\Exception $exception) {
            return response()->json([
                'error' => true,
                'message' => $exception->getMessage()
            ], 404);
        }
    }


    /**
     * Remove the specified resource from storage.
     *
     * @param  int $id
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id)
    {
        try {
            $user = auth()->user();
            $profile = Profile::where('idPerfil', $id)->first();
            $response = response()->json($this->service->delete($id), 204);
            Logs::writeLog($user->stApelido, "PERFIS", 'O OPERADOR [' . $user->idOperador . '-' . $user->stApelido . '] DESATIVOU O PERFIL [' . $id . ' - ' . $profile->stDescricao . ']');

            return $response;

        } catch (\Exception $exception) {
            return response()->json([
                'error' => true,
                'message' => $exception->getMessage()
            ], 404);
        }
    }
}
