<?php

namespace App\Http\Controllers\Mobyo;

use Illuminate\Http\Request;

use App\Http\Requests;
use Prettus\Validator\Contracts\ValidatorInterface;
use Prettus\Validator\Exceptions\ValidatorException;
use App\Http\Requests\MobyoPaymentCreateRequest;
use App\Http\Requests\MobyoPaymentUpdateRequest;
use App\Repositories\Mobyo\MobyoPaymentRepository;
use App\Validators\Mobyo\MobyoPaymentValidator;

/**
 * Class MobyoPaymentsController.
 *
 * @package namespace App\Http\Controllers\\Mobyo;
 */
class MobyoPaymentsController extends Controller
{
    /**
     * @var MobyoPaymentRepository
     */
    protected $repository;

    /**
     * @var MobyoPaymentValidator
     */
    protected $validator;

    /**
     * MobyoPaymentsController constructor.
     *
     * @param MobyoPaymentRepository $repository
     * @param MobyoPaymentValidator $validator
     */
    public function __construct(MobyoPaymentRepository $repository, MobyoPaymentValidator $validator)
    {
        $this->repository = $repository;
        $this->validator  = $validator;
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $this->repository->pushCriteria(app('Prettus\Repository\Criteria\RequestCriteria'));
        $mobyoPayments = $this->repository->all();

        if (request()->wantsJson()) {

            return response()->json([
                'data' => $mobyoPayments,
            ]);
        }

        return view('mobyoPayments.index', compact('mobyoPayments'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  MobyoPaymentCreateRequest $request
     *
     * @return \Illuminate\Http\Response
     *
     * @throws \Prettus\Validator\Exceptions\ValidatorException
     */
    public function store(MobyoPaymentCreateRequest $request)
    {
        try {

            $this->validator->with($request->all())->passesOrFail(ValidatorInterface::RULE_CREATE);

            $mobyoPayment = $this->repository->create($request->all());

            $response = [
                'message' => 'MobyoPayment created.',
                'data'    => $mobyoPayment->toArray(),
            ];

            if ($request->wantsJson()) {

                return response()->json($response);
            }

            return redirect()->back()->with('message', $response['message']);
        } catch (ValidatorException $e) {
            if ($request->wantsJson()) {
                return response()->json([
                    'error'   => true,
                    'message' => $e->getMessageBag()
                ]);
            }

            return redirect()->back()->withErrors($e->getMessageBag())->withInput();
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int $id
     *
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $mobyoPayment = $this->repository->find($id);

        if (request()->wantsJson()) {

            return response()->json([
                'data' => $mobyoPayment,
            ]);
        }

        return view('mobyoPayments.show', compact('mobyoPayment'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int $id
     *
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $mobyoPayment = $this->repository->find($id);

        return view('mobyoPayments.edit', compact('mobyoPayment'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  MobyoPaymentUpdateRequest $request
     * @param  string            $id
     *
     * @return Response
     *
     * @throws \Prettus\Validator\Exceptions\ValidatorException
     */
    public function update(MobyoPaymentUpdateRequest $request, $id)
    {
        try {

            $this->validator->with($request->all())->passesOrFail(ValidatorInterface::RULE_UPDATE);

            $mobyoPayment = $this->repository->update($request->all(), $id);

            $response = [
                'message' => 'MobyoPayment updated.',
                'data'    => $mobyoPayment->toArray(),
            ];

            if ($request->wantsJson()) {

                return response()->json($response);
            }

            return redirect()->back()->with('message', $response['message']);
        } catch (ValidatorException $e) {

            if ($request->wantsJson()) {

                return response()->json([
                    'error'   => true,
                    'message' => $e->getMessageBag()
                ]);
            }

            return redirect()->back()->withErrors($e->getMessageBag())->withInput();
        }
    }


    /**
     * Remove the specified resource from storage.
     *
     * @param  int $id
     *
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $deleted = $this->repository->delete($id);

        if (request()->wantsJson()) {

            return response()->json([
                'message' => 'MobyoPayment deleted.',
                'deleted' => $deleted,
            ]);
        }

        return redirect()->back()->with('message', 'MobyoPayment deleted.');
    }
}
