<?php

namespace App\Http\Controllers;

use App\Entities\Logs;
use App\Http\Controllers\Controller;
use App\Http\Requests\Nati\StockCreateRequest;
use App\Http\Requests\Nati\StockUpdateRequest;
use App\Services\Nati\StockService;
use App\Services\Nati\AdjustStocksService;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use App\Services\Nati\Service;
use Prettus\Validator\Exceptions\ValidatorException;

class StocksController extends Controller
{

    protected $service;
    protected $baseService;
    protected $stocksService;

    public function __construct(StockService $service, Service $baseService, AdjustStocksService $stocksService)
    {
        $this->service = $service;
        $this->baseService = $baseService;
        $this->stocksService = $stocksService;
    }

    /**
     * Display a listing of the resource.
     *
     * @return JsonResponse
     */
    public function index()
    {
        try
        {
            $estoque = $this->service->all();
            return response()->json($estoque);
        }
        catch(\Exception $exception)
        {
            return response()->json([
                'error' => true,
                'message' => $exception->getMessage()
            ], 400);
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  StockCreateRequest $request
     *
     * @return JsonResponse
     */
    public function store(StockCreateRequest $request)
    {
        try {
            $estoque = $this->service->create($request->all());

            $estoque = [
                'message' => 'Estoque created.',
                'data'    => $estoque->toArray(),
            ];
            return response()->json($estoque, 201);

        }catch (ModelNotFoundException $e) {
            return response()->json([
                'error'   => true,
                'message' => $e->getMessageBag()
            ], 400);
        }
        catch (ValidatorException $e) {
            return response()->json([
                'error'   => true,
                'message' => $e->getMessageBag()
            ], 404);
        }
    }


    /**
     * Display the specified resource.
     *
     * @param  int $id
     *
     * @return JsonResponse
     */
    public function show($id)
    {
        try
        {
            $estoque = $this->service->getById($id);
            return response()->json($estoque);
        }
        catch(\Exception $exception)
        {
            return response()->json([
                'error' => true,
                'message' => $exception->getMessage()
            ], 404);
        }
    }


    /**
     * Update the specified resource in storage.
     *
     * @param  StockUpdateRequest $request
     * @param  string            $id
     *
     * @return JsonResponse
     */
    public function update(StockUpdateRequest $request, $id)
    {
        try {
            $estoque = $this->service->update($request->all(), $id);
            return response()->json([
                'message' => 'Estoque Update',
                'data' => $estoque
            ],201);

        } catch (ModelNotFoundException $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessage()
            ], 400);

        } catch (ValidatorException $e) {
            return response()->json([
                'error'   => true,
                'message' => $e->getMessageBag()
            ], 404);
        }
    }

    /**
     * Parses XML string to JSON.
     *
     * @param Request $request
     *
     * @return JsonResponse
     */
    public function uploadXml(Request $request)
    {
        $req = $request->all();

        try {
            $response = $this->baseService->xmlToArray(simplexml_load_string($req['xml']));
            $response = $this->baseService->parseXmlArray($response);
            return response()->json(['message' => 'Success', 'data' => $response]);

        } catch (\Exception $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessage()
            ], 400);
        }
    }


    /**
     * Remove the specified resource from storage.
     *
     * @param  int $id
     *
     * @return JsonResponse
     */
    public function destroy($id)
    {
        try {
            $aliquota = $this->service->delete($id);

            $response = [
                'message' => 'Estoque deleted.',
                'data'    => $aliquota,
            ];
            return response()->json($response);

        } catch (\Exception $e) {
            return response()->json([
                'error'   => true,
                'message' => $e->getMessage()
            ], 400);
        }
    }

    public function getStocks()
    {
        try {
            $estoques = $this->stocksService->getStocks();
            return response()->json($estoques);

        } catch (\Exception $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessage()
            ], 400);
        }
    }

    public function filterProducts(Request $request)
    {
        try {
            $products = $this->stocksService->filterProducts($request);
            return response()->json($products);
        } catch (\Exception $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessage()
            ], 400);
        }
    }

    public function adjust(Request $request)
    {
        try{
            $user = auth()->user();
            $response = $this->stocksService->adjustStock($request->all());
            Logs::writeLog($user->stApelido, "ESTOQUE", 'O OPERADOR [' . $user->stApelido . '] FEZ AJUSTE DE ESTOQUE');
            return response()->json($response);
        } catch (\Exception $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessage()
            ], 400);
        }
    }
}
