<?php

namespace App\Services\Nati;

use App\Entities\Nati\Sector;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Str;
use App\Repositories\Nati\SectorRepository;
use App\Validators\Nati\SectorValidator;
use Prettus\Validator\Contracts\ValidatorInterface;
use Carbon\Carbon;

class SectorService
{
    private $repository;
    private $validator;

    public function __construct(SectorRepository $repository, SectorValidator $validator)
    {
        $this->repository = $repository;
        $this->validator = $validator;
    }

    public function getById($id)
    {
        return $this->repository->withCount('produtos')->find($id);
    }

    public function all($request)
    {
        $key = md5(json_encode($request));
        if (Cache::tags('ListaSetores')->has($key)) {
            return Cache::tags('ListaSetores')->get($key);
        } else {
            $response = Sector::where('flAtivo', true)
                ->where(function ($query) use ($request) {
                    if (isset($request['filter']) and strlen($request['filter']) > 0) {
                        $query->where('stSetor', 'LIKE', '%' . $request['filter'] . '%');
                    }
                })
                ->withCount('produtos')
                ->paginate($request['limit'], $request['page']);

            Cache::tags('ListaSetores')->put($key, $response, 1440);
            return $response;
        }
    }

    public function getAll()
    {
        if (Cache::tags('ListaSetores')->has('all')) {
            return Cache::tags('ListaSetores')->get('all');
        } else {
            $response = $this->repository->orderBy('stSetor','asc')
                ->findWhere(['flAtivo' => true]);
            Cache::tags('ListaSetores')->put('all', $response, 1440);
            return $response;
        }
    }

    public function create($request)
    {
        $this->validator->with($request)->passesOrFail(ValidatorInterface::RULE_CREATE);

        $setor = [
            "idSetor" => $this->getNextId(),
            "stSetor" => $request['stSetor'],
            "flAtivo" => true,
            "dtAlteracao" => Carbon::now(),
        ];

        return $this->repository->create($setor);
    }

    public function update($request, $id)
    {
        $this->validator->with($request)->passesOrFail(ValidatorInterface::RULE_UPDATE);

        $request['dtAlteracao'] = Carbon::now();

        return $this->repository->update($request, $id);
    }

    public function delete($id)
    {
        return $this->repository->update(['flAtivo' => false, 'dtAlteracao' =>Carbon::now()], $id);
    }

    private function getNextId()
    {
        return Sector::max('idSetor') + 1;
    }

    public function getOrCreate($setor)
    {
        $cat = $this->repository->where('stSetor', strtoupper($setor))->first();
        if($cat){
            return $cat;
        }else{
            return $this->create(['stSetor' => strtoupper($setor)]);
        }
    }
}
