<?php

namespace App\Http\Controllers;

use App\Entities\Logs;
use App\Entities\RespostaObservacao;
use App\Services\Nati\RespostaObservacaoService;
use Illuminate\Http\Request;
use Prettus\Validator\Exceptions\ValidatorException;
use App\Http\Requests\RespostaObservacaoCreateRequest;
use App\Repositories\RespostaObservacaoRepository;
use App\Validators\RespostaObservacaoValidator;

/**
 * Class RespostaObservacaosController.
 *
 * @package namespace App\Http\Controllers;
 */
class RespostaObservacoesController extends Controller
{
    /**
     * @var RespostaObservacaoRepository
     */
    protected $repository;

    /**
     * @var RespostaObservacaoValidator
     */
    protected $validator;

    /**
     * @var RespostaObservacaoService
     */
    protected $service;

    /**
     * RespostaObservacoesController constructor.
     *
     * @param RespostaObservacaoRepository $repository
     * @param RespostaObservacaoValidator $validator
     */
    public function __construct(RespostaObservacaoRepository $repository, RespostaObservacaoValidator $validator,  RespostaObservacaoService $service)
    {
        $this->repository = $repository;
        $this->validator  = $validator;
        $this->service = $service;
    }

    public function getAll(Request $request)
    {
        try {
            return response()->json($this->service->getAll($request->all()));
        } catch (\Exception $exception) {
            return response()->json([
                'error' => true,
                'message' => $exception->getMessage()
            ], 400);
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  RespostaObservacaoCreateRequest $request
     *
     * @return \Illuminate\Http\Response
     *
     * @throws \Prettus\Validator\Exceptions\ValidatorException
     */
    public function store(RespostaObservacaoCreateRequest $request)
    {
        try {
            $respostaObservacao = $this->service->store($request->all());
            $user = auth()->user();
            Logs::writeLog($user->stApelido, "OBSERVACOES", 'O USUARIO [' . $user->stApelido . '] CADASTROU A OBSERVACAO [' . $respostaObservacao->descricao . ']');

            return response()->json([
                'message' => 'Observação cadastrada',
                'data' => $respostaObservacao->toArray(),
            ], 200);
        } catch (ValidatorException $e) {
            if ($request->wantsJson()) {
                return response()->json([
                    'error'   => true,
                    'message' => $e->getMessage()
                ], 500);
            }

            return redirect()->back()->withErrors($e->getMessageBag())->withInput();
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int $id
     *
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        try {
            return $this->service->show($id);
        } catch (\Exception $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessage()
            ], 400);
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int $id
     *
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        try {
            $oldNameObs = RespostaObservacao::where('idObservacao', $id)->first();
            $user = auth()->user();
            $response = $this->service->update($request->all(), $id);
            Logs::writeLog($user->stApelido, "OBSERVACOES", 'O USUARIO [' . $user->stApelido . '] ALTEROU O NOME DA OBSERVACAO [' . $oldNameObs->stObservacao . '] ' . 'PARA [' . $response->stObservacao . ']');
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessage()
            ], 404);
        } catch (ValidatorException $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessageBag()
            ], 400);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int $id
     *
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        try {
            $user = auth()->user();
            $response = $this->service->delete($id);
            Logs::writeLog($user->stApelido, "OBSERVACOES", 'O OPERADOR [' . $user->stApelido . '] DESATIVOU A OBSERVACAO [' . $response->stObservacao . ']');
            return response()->json([

            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessage()
            ], 400);
        }
    }

    public function filter(string $data)
    {
        try{
            return response()->json($this->service->filter($data));
        } catch(\Exception $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessage()
            ], 400);
        }
    }
}
