<?php
/**
 * Created by PhpStorm.
 * User: ARCNET DEV
 * Date: 13/11/2017
 * Time: 16:50
 */

namespace App\Services;

use App\Validators\Nati\FichaPagamentoValidator;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\Collection;

class PaymentFormService
{
    protected $repository;
    protected $validator;
    protected $temp;
    protected $clienteService;

    public function __construct
    (
        FichaPagamentoRepository $repository,
        FichaPagamentoValidator $validator,
        FichaRecebimentoTempRepository $temp,
        ClienteService $clienteService
    ){
        $this->repository = $repository;
        $this->validator = $validator;
        $this->temp = $temp;
        $this->clienteService = $clienteService;
    }

    public function getFichaPagamento($temp = false)
    {
        return $this->getRepository($temp)->all();
    }

    public function getFichaPagamentoById($idAbertura, $idVenda, $temp = false)
    {
        if (is_null($idVenda)) {
            return $this->getRepository($temp)->with(['formaPagamento'])
                ->findWhere(['idAbertura' => $idAbertura]);
        }
        else return $this->getRepository($temp)->with(['formaPagamento'])
            ->findWhere(['idAbertura' => $idAbertura, 'idVenda' => $idVenda]);
    }

    public function getContaAssinada($idAbertura, $idVenda, $temp = false)
    {
        return $this->getFichaPagamentoById($idAbertura, $idVenda, $temp)->filter(function($item) {
            $isContaAssinada = [12, 14];
            return in_array($item->formaPagamento->tipo()->first()->idTipoForma, $isContaAssinada);
        });
    }

    /** Returns detailed infomation of sales of the specified FormaPagamento
     *
     * @param integer $idAbertura
     * @param integer $idForma,
     * @param integer $page,
     * @param boolean $temp
     * @return LengthAwarePaginator
     * */
    public function getDetalhesFormaPagamento($idAbertura, $idForma, $page = 1, $temp = false)
    {
        $result = $this->getFichaPagamentoById($idAbertura, null, $temp)->filter(function($ficha) use($idForma) {
            return $ficha->idFormaPagamento == $idForma;
        });
        // Configurações para o paginate
        $paginate = 10;
        $offset = ($page * $paginate) - $paginate;
        $total = intval(count($result) / $paginate);
        $total = $total == 0 ? 1 : $total;
        $itemsForCurrentPage = array_slice($result->toArray(), $offset, $paginate, true);
        $fichas = [];

        foreach ($itemsForCurrentPage as $ficha) {
            // É necessário que a colection de fichas contenha Models de ficha e não arrays
            // para o resource poder fazer o parse corretamente da collection
            $ficha = $temp ? new FichaRecebimentoTemp($ficha) : new FichaPagamento($ficha);
            // Traz os relacionamentos
            $ficha = $ficha->load(['venda', 'formaPagamento']);

            $ficha->cliente = $this->clienteService->getById($ficha->venda->idCliente)->stCliente;
            $ficha->venda->isTemp = $temp;
            array_push($fichas, $ficha);
        }
        return new LengthAwarePaginator($fichas, $total, $paginate, $page);
    }

    public function getTotalContaAssinada(Collection $contaAssinada)
    {
        return $contaAssinada->reduce(function($total, $data) {
            return $total + floatval($data->vrTotal);
        },0);
    }

    public function getTotalFormaPagamento(Collection $vendas, $temp = false)
    {
        $result = [];
        foreach($vendas as $venda) {
            $fichaPagamento = $this->getFichaPagamentoById($venda->idAbertura, $venda->idVenda, $temp)
                ->filter(function($item) {
                    return !in_array($item->formaPagamento->tipo()->first()->idTipoForma, [12, 14]);
                });

            foreach ($fichaPagamento as $ficha) {
                $exists = false;
                $formaPagamento = $ficha->formaPagamento;
                foreach($result as $forma) {
                    if ($forma['nome'] == $formaPagamento->stFormaPagamento) {
                        $forma['sistema'] += $ficha->vrFormaPagamento;
                        $exists = true;
                    }
                }
                if ($exists == false) {
                    array_push($result, collect([
                        'id' => $ficha->idFormaPagamento,
                        'nome' => $formaPagamento->stFormaPagamento,
                        'sistema' => $ficha->vrFormaPagamento]));
                }
            }
        }
        return $result;
    }

    public function getRepository($isTemp)
    {
        return $isTemp ? $this->temp : $this->repository;
    }

    public function update($idFicha, $idVenda, $idAbertura, $temp, $data)
    {
        $ficha = $this->getRepository($temp)
            ->findWhere(['idFicha' => $idFicha, 'idVenda' => $idVenda, 'idAbertura' => $idAbertura])->first();

        $ficha->fill($data);
        $ficha->save();
        return $ficha;
    }
}
