<?php

namespace App\Entities;

use App\Entities\Nati\CustomerPayment;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Prettus\Repository\Contracts\Transformable;
use Prettus\Repository\Traits\TransformableTrait;
use Database\Factories\ManagerFactory;

class Manager extends Model implements Transformable
{
    use TransformableTrait, HasFactory;
    protected $table = 'cx_Gerenciador';
    protected $primaryKey = 'idAbertura';
    public $incrementing = false;
    public $timestamps = false;

    protected $fillable = [
        'idAbertura',
        'idCaixa',
        'idOperador',
        'dtAbertura',
        'dtFechamento',
        'vrAbertura',
        'vrFechamento',
        'flAberto',
        'vrTrocoEncerramento',
        'flConferencia',
        'nrClientes',
        'idOrigem',
        'idDestino',
        'stOBS',
        'dtAlteracao',
        'uuid'
    ];

    protected $casts = [
        'idCaixa' => 'integer',
        'idAbertura' => 'integer',
        'flAberto' => 'boolean',
        'flConferencia'=> 'boolean',
        'vrAbertura'=> 'float',
        'vrTrocoEncerramento'=> 'float',
        'vrFechamento'=> 'float',
    ];

    public static function boot()
    {
        parent::boot();
        self::creating(function ($model) {
            $model->uuid = (string)Str::uuid();
        });
    }

    //Get Attribute
    public function getTotalAttribute()
    {
        $total = $this->vendasTemp()->where('flAtivo', true)->sum('vrTotal');
        $total += $this->cliPagamento()->where('flAtivo', true)->sum('vrPagamento');

        if ($total) {
            return $total;
        }

        return 0;
    }

    public function getQuantityTotalAttribute()
    {
        return $this->vendasTemp()->where('active', true)->count();
    }

    //Relationships
    public function operador()
    {
        return $this->belongsTo(User::class, 'idOperador', 'idOperador');
    }

    public function sangrias()
    {
        return $this->hasMany(Sangria::class, 'idAbertura', 'idAbertura');
    }

    public function despesas()
    {
        return $this->hasMany(Expense::class, 'idAbertura', 'idAbertura');
    }

    public function voucher()
    {
        return $this->hasMany(Voucher::class, 'idAbertura', 'idAbertura');
    }

    public function repiques()
    {
        return $this->hasMany(Repique::class, 'idAbertura', 'idAbertura');
    }

    public function cliPagamento()
    {
        return $this->hasMany(CustomerPayment::class, 'idAbertura', 'idAbertura');
    }

    //Relationships TEMP [antes de fechar pdv]
    public function vendasTemp()
    {
        return $this->hasMany(SaleTemp::class, 'idAbertura', 'idAbertura');
    }

    public function recebimentos()
    {
        return $this->hasMany(ReceiptForm::class, 'idAbertura', 'idAbertura');
    }

    public function itensTemp()
    {
        return $this->hasMany(CxItemsTemp::class, 'idAbertura', 'idAbertura');
    }

    //Relationships [depois de fechar pdv]
    public function pagamentos()
    {
        return $this->hasMany(PaymentMethod::class, 'idAbertura', 'idAbertura');
    }

    public function vendas()
    {
        return $this->hasMany(Sale::class, 'idAbertura', 'idAbertura');
    }

    public function detalhes()
    {
        return $this->hasMany(ManagerDetails::class, 'idAbertura', 'idAbertura');
    }

    public function historico()
    {
        return $this->belongsTo(ClosingHistory::class, 'idAbertura', 'idAbertura');
    }

    public function needToClose(){
        $noHour = Carbon::parse($this->dtAbertura)->format("Y-m-d");
        return Carbon::parse($noHour)->diffInDays(Carbon::now()) > 0;
    }

    //Functions

    // Todas vendas e deposito de conta assinada ATIVA em cartões
    // Agrupando valores
    public function cardPayments()
    {
        $customer_payments = DB::table('cli_Pagamento as CP')
            ->select(DB::raw("CP.idAbertura as id_open, CP.idFormaPagamento as id, FP.idTipo as id_type_payment, FP.stFormaPagamento as name"),
                DB::raw('count(*) as quantity'),
                DB::raw('count(CASE CP.flConferido WHEN 1 THEN 1 ELSE NULL END) as quantity_conferred'),
                DB::raw('sum(CASE WHEN CP.flConferido = 1 THEN CP.vrPagamento ELSE 0 END) as value_conferred'),
                DB::raw('sum(CP.vrPagamento) as value'))
            ->join('cx_FormasPagamento as FP', 'CP.idFormaPagamento', 'FP.idFormaPagamento')
            ->where('CP.idAbertura', $this->idAbertura)
            ->where('FP.idTipo', 2)
            ->where('CP.flAtivo', true)
            ->groupBy('CP.idAbertura', 'CP.idFormaPagamento', 'FP.idTipo', 'FP.stFormaPagamento');

        $receiveds = DB::table('cx_FichaRecebimentos_Temp as FR')
            ->select(DB::raw("FR.idAbertura as id_open, FR.idFormaPagamento as id, FP.idTipo as id_type_payment, FP.stFormaPagamento as name"),
                DB::raw('count(DISTINCT FR.idVenda) as quantity'),
                DB::raw('count(CASE FR.flConferido WHEN 1 THEN 1 ELSE NULL END) as quantity_conferred'),
                DB::raw('sum(CASE WHEN FR.flConferido = 1 THEN FR.vrFormaPagamento ELSE 0 END) as value_conferred'),
                DB::raw('sum(FR.vrFormaPagamento) as value'))
            ->join('cx_FormasPagamento as FP', 'FR.idFormaPagamento', 'FP.idFormaPagamento')
            ->where('FR.idAbertura', $this->idAbertura)
            ->where('FP.idTipo', 2)
            ->where('FR.flAtivo', true)
            ->groupBy('FR.idAbertura', 'FR.idFormaPagamento', 'FP.idTipo', 'FP.stFormaPagamento')
            ->union($customer_payments);

        return DB::table(DB::raw("({$receiveds->toSql()}) as x"))
            ->select(DB::raw('id_open, id, id_type_payment, name'),
                DB::raw('sum(quantity) as quantity'),
                DB::raw('sum(quantity_conferred) as quantity_conferred'),
                DB::raw('sum(value_conferred) as value_conferred'),
                DB::raw('sum(value) as value'))
            ->mergeBindings($receiveds)
            ->groupBy('id_open', 'id', 'id_type_payment', 'name')
            ->orderBy('value', 'DESC')
            ->get();
    }

    protected static function newFactory()
    {
        return ManagerFactory::new();
    }
}
