<?php


namespace App\Services\Sales;

use App\Entities\CxItemsTemp;
use App\Entities\Delivery\Order;
use App\Entities\ReceiptForm;
use App\Entities\ReceiptFormTemp;
use App\Entities\Sale;
use App\Entities\SalesItensView;
use App\Entities\vwSalesItemsReport;
use App\Entities\SalesView;
use App\Entities\SaleTemp;
use App\Events\DeliverySaleCreated;
use App\Repositories\SaleTempRepository;
use App\Services\ManagerService;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class SalesService
{
    protected $managerService;
    protected $abertura;
    protected $config;
    protected $repository;

    public function __construct(SaleTempRepository $repository, ManagerService $managerService)
    {
        $this->repository = $repository;
        $this->managerService = $managerService;
        $this->config = (object)app('config')->get('ifood');
    }

    public function get($request)
    {
        $fd = Carbon::createFromFormat('d/m/Y', $request['start'])->toDateString();
        $first_date = $fd . ' 00:00:00';
        $ld = Carbon::createFromFormat('d/m/Y', $request['end'])->toDateString();
        $last_date = $ld . ' 23:59:59';
        $order = !isset($request['order']) ? 'DESC' : Str::upper($request['order']);
        $sort = $request['sort'] == '' ? 'dtData' : $request['sort'];

        $q = SalesView::select('*')
            ->whereBetween(DB::raw('cast(dtData AS DATE)'), [$first_date, $last_date]);

        if ($request['active'] == 'false') {
            $q->where('flAtivo', true);
        }

        $q->orderBy($sort, $order);

        if ($sort == 'idVenda') {
            $q->orderBy('idAbertura', $order);
        }

        $sale = $q->paginate($request['limit'], $request['page']);

        return $sale;
    }

    public function getVenda($idAbertura, $idVenda)
    {
        $sale = SalesView::where('idAbertura', $idAbertura)->where('idVenda', $idVenda)->first();
        $sale['itens'] = SalesItensView::where('idAbertura', $idAbertura)->where('idVenda', $idVenda)->get();

//        $sale = Sale::with(['itens', 'itens.produto', 'itens.produto.medida', 'customer', 'abertura', 'abertura.operador'])
//            ->where('idAbertura', $idAbertura)
//            ->where('idVenda', $idVenda)->first();
        $sale->pagamentos = ReceiptForm::with('formaPagamento')
            ->where('idAbertura', $idAbertura)
            ->where('idVenda', $idVenda)->get();

        return $sale;
    }


    public function fromDelivery($idOrder)
    {
        try {
            DB::beginTransaction();
            $this->abertura = $this->managerService->getByPos($this->config->id_caixa_padrao, $this->config->user_id);
            $order = Order::find($idOrder);

            if (!$order) {
                throw new \Exception('Order not found');
            }
            if (isset($order->idVenda) && isset($order->idAbertura)) {
                throw new \Exception('Este pedido já foi processado');
            }

            $sale = new SaleTemp([
                'idVenda' => $this->nextId(),
                'idAbertura' => $this->abertura->idAbertura,
                'vrTotal' => $order->total(),
                'flNotaFiscal' => isset($order->chaveFiscal) ? true : false,
                'idCliente' => $order->idCliente,
                'idOperadorAutorizador' => 999,
                'flDelivery' => true,
                'idAtendimentoDelivery' => $order->idAtendimento,
                'stCPF' => isset($order->stCpf) ? $order->stCpf : '',
                'chave_fiscal' => $order->chaveFiscal,
            ]);
            $this->abertura->vendasTemp()->save($sale);

            foreach ($order->items()->get() as $item) {

                $it = new CxItemsTemp([
                    'idItem' => $item->nrItem,
                    'idVenda' => $sale->idVenda,
                    'idAbertura' => $sale->idAbertura,
                    'idProduto' => $item->idProduto,
                    'dtData' => Carbon::now(),
                    'nrQuantidade' => $item->nrQuantidade,
                    'vrCusto' => 0,
                    'vrUnitario' => $item->vrUnitario,
                    'vrDesconto' => $item->desconto,
                    'flAtivo' => $item->flAtivo,
                    'idComanda' => 0,
                    'idCodProduto' => $item->codigo,
                    'vrTotal' => $item->vrTotal,
                    'vrTotalCusto' => 0,
                    'idOperadorAutorizador' => 999,
                    'stAtendente' => $item->stOperador,
                    'flEstoqueBaixado' => false,
                    'flPromocao' => false,
                    'dtAlteracao' => Carbon::now(),
                ]);

                $sale->itens()->save($it);
            }

            foreach ($order->payments()->get() as $pay) {
                $payment = new ReceiptFormTemp();
                $payment->idFicha = ReceiptFormTemp::nextId($sale->idAbertura, $sale->idVenda);
                $payment->idVenda = $sale->idVenda;
                $payment->idAbertura = $sale->idAbertura;
                $payment->idFormaPagamento = $pay->idFormaPagamento;
                $payment->vrFormaPagamento = $pay->vrFormaPagamento;
                $payment->flTroco = false;

                $sale->paymentsTemp()->save($payment);
            }

            $order->idVenda = $sale->idVenda;
            $order->idAbertura = $sale->idAbertura;
            $order->stStatus = "E";
            $order->flPago = true;
            $order->save();

            DB::commit();
            return $sale;
        } catch (\Exception $e) {
            DB::rollBack();
        }
    }

    protected function nextId()
    {
        return SaleTemp::where('idAbertura', $this->abertura->idAbertura)
                ->max('idVenda') + 1;
    }

    public function setChaveFiscal($order)
    {
        $sale = $this->repository->findWhere([
            'idVenda' => $order->idVenda,
            'idAbertura' => $order->idAbertura
        ])->first();

        $sale->flNotaFiscal = true;
        $sale->chave_fiscal = $order->chaveFiscal;
        $sale->save();
    }

    public function getSalesItemsReport($request){
        $fd = Carbon::createFromFormat('d/m/Y', $request['start'])->toDateString();
        $first_date = $fd . ' 00:00:00';
        $ld = Carbon::createFromFormat('d/m/Y', $request['end'])->toDateString();
        $last_date = $ld . ' 23:59:59';
        $order = !isset($request['order']) ? 'DESC' : Str::upper($request['order']);
        $sort = $request['sort'] == '' ? 'dtData' : $request['sort'];

        $q = vwSalesItemsReport::select(
            'idProduto',
            'idCodProduto',
            'stProduto',
            'idAliquota',
            'stAliquota',
            'idCategoria',
            'stCategoria',
            'idSetor',
            'stSetor',
            'idMedida',
            'stMedida',
            'nrNCM',
            'flAtivo',
            DB::raw('SUM(nrQuantidade) as nrQuantidade'),
            DB::raw('SUM(vrUnitario) as vrUnitario'),
            DB::raw('SUM(vrDesconto) as desconto'),
            DB::raw('SUM(vrTotal) as vrTotal'),
            DB::raw('SUM(total) as total')
        )
            ->whereBetween(DB::raw('cast(dtData AS DATE)'), [$first_date, $last_date])
            ->where('flAtivo', true);


        if ($request['sector']) {
            $q->where('idSetor', $request['sector']);
        }

        if ($request['category']) {
            $q->where('idCategoria', $request['category']);
        }

        if ($request['aliquota']) {
            $q->where('idAliquota', $request['aliquota']);
        }

        if ($request['measure']) {
            $q->where('idMedida', $request['measure']);
        }

        $q->groupBy('idProduto', 'idCodProduto', 'stProduto', 'idAliquota', 'stAliquota', 'idCategoria', 'stCategoria', 'idSetor', 'stSetor', 'flAtivo', 'idMedida', 'stMedida', 'nrNCM');

        $q->orderBy($sort, $order);
        $result = [];
        $result['sales'] = $q->paginate($request['limit'], $request['page']);

        $result['canceledItems'] = DB::table('vw_sales_items_report')
                                            ->select(DB::raw('count(*) as qtd'))
                                            ->whereBetween(DB::raw('cast(dtData AS DATE)'), [$first_date, $last_date])
                                            ->where('flAtivo', false)
                                            ->get();

        $result['totalOfSales'] = DB::table(function ($query) {
                                                $query->selectRaw('idVenda, idAbertura, dtData')
                                                    ->from('vw_sales_items_report')
                                                    ->where('flAtivo', true)
                                                    ->groupBy('idVenda', 'idAbertura', 'dtData');
                                                }, 'x')
                                                ->select(DB::raw('count(*) as qtdSales'))
                                                ->whereBetween(DB::raw('cast(dtData AS DATE)'), [$first_date, $last_date])
                                                ->get();

        return $result;
    }

    public function getSalesReport($request){
        $fd = Carbon::createFromFormat('d/m/Y', $request['start'])->toDateString();
        $first_date = $fd . ' 00:00:00';
        $ld = Carbon::createFromFormat('d/m/Y', $request['end'])->toDateString();
        $last_date = $ld . ' 23:59:59';
        $order = !isset($request['order']) ? 'DESC' : Str::upper($request['order']);
        $sort = $request['sort'] == '' ? 'dtData' : $request['sort'];

        $sales = SalesView::select(
            'idVenda',
            'idAbertura',
            'idCliente',
            'stCliente',
            'flAtivo',
            'vrDesconto',
            'vrAcrescimo',
            'vrTotal',
            'dtData',
            'flNotaFiscal',
            'idCaixa',
            'idOperador',
            'stApelido',
            'chave_fiscal',
            'vrTotalBruto'
        )
        ->whereBetween(DB::raw('cast(dtData AS DATE)'), [$first_date, $last_date])
        ->where('flAtivo', true);

        if ($request['cashier']) {
            $sales->where('idOperador', $request['cashier']);
        }

        if ($request['pdv']) {
            $sales->where('idCaixa', $request['pdv']);
        }

        $sales->orderBy($sort, $order);

        $salesPaginate = [];

        $salesPaginate = $sales->paginate($request['limit'], $request['page']);

        return $salesPaginate;
    }
}
