<?php

namespace App\Http\Controllers\Financeiro;

use App\Http\Controllers\Controller;
use App\Repositories\ExpenseRepository;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use App\Entities\Expense;

class DespesasController extends Controller
{
    protected $repository;

    public function __construct(ExpenseRepository $repository)
    {
        $this->repository = $repository;
    }

    public function index()
    {
        try {

            $despesas = $this->repository->paginate($limit = null, $columns = ['*']);

            return response()->json($despesas, 200);

        } catch (\Exception $exception) {
            return response()->json([
                'error' => true,
                'message' => $exception->getMessage()
            ], 500);
        }
    }

    public function getDespesas(Request $request)
    {
        try {
            $startDate = Carbon::createFromFormat('d/m/Y', $request['start'])->toDateString() . ' 00:00:00';
            $endDate = Carbon::createFromFormat('d/m/Y', $request['end'])->toDateString() . ' 23:59:59';
            $order = !isset($request['order']) ? 'DESC' : Str::upper($request['order']);
            $sort = $request['sort'] == '' ? 'dtData' : $request['sort'];
            return Expense::select('idDespesa', 'idAbertura', 'stDescricao', 'dtData', 'vrDespesa', 'idFormaPagamento', 'flCompensado', 'flAtivo', 'dtAlteracao', 'uuid')
                ->with('gerenciador')
                ->with('formaPagamento')
                ->where('dtData', '>=', $startDate)
                ->where('dtData', '<=', $endDate)
                ->where('flAtivo', true)
                ->orderBy($sort, $order)
                ->paginate($request['limit'], $request['page']);

        } catch (\Throwable $th) {
            //throw $th;
        }
    }

    public function pending(Request $request)
    {
        try {
            $start_date = Carbon::createFromFormat('d/m/Y', $request['de'])->toDateString();
            $end_date = Carbon::createFromFormat('d/m/Y', $request['ate'])->toDateString();


            $despesas = $this->repository
                ->with('formaPagamento')
                ->findWhere([
                    'is_compensado' => false,
                    'active' => true,
                    ['date', '>=', $start_date . ' 00:00:00'],
                    ['date', '<=', $end_date . ' 23:59:59'],
                ]);

            return response()->json($despesas, 200);
        } catch (\Exception $exception) {
            return response()->json([
                'error' => true,
                'message' => $exception->getMessage()
            ], 500);
        }
    }

    public function delete(Request $request)
    {
        try {
            $response = $this->repository->findWhere([
                'id' => $request->id,
                'id_open' => $request->id_open
            ])->first()->update(['active' => false]);

            return response()->json([
                'message' => 'Conta a pagar removida',
                'data' => $response
            ]);
        } catch (\Exception $exception) {
            return response()->json(['error' => true,
                'message' => $exception->getMessage()], 500);
        }
    }
}
