<?php

namespace App\Http\Controllers;

use App\Http\Resources\Nati\OrderTabResource;
use App\Services\Nati\OrderTabService;
use Exception;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Routing\Controller;
use Illuminate\Validation\UnauthorizedException;

class OrderTabController extends Controller
{
    private $service;

    public function __construct(OrderTabService $service)
    {
        $this->service = $service;
    }

    /**
     * @param Request $request
     * @return JsonResponse
     */
    public function getAll(Request $request)
    {
        try {
            return response()->json($this->service->getAll($request->all()));
        } catch (Exception $exception) {
            return response()->json([
                'error' => true,
                'message' => $exception->getMessage()
            ], 404);
        }
    }

    /**
     * @param string $tabId
     * @return OrderTabResource|JsonResponse
     */
    public function show(string $tabId)
    {
        try {
            $response = $this->service->show($tabId);
            if ($response != null) {
                return new OrderTabResource($response);
            }
            return response()->json(["OrderTab does not exist"], 404);
        } catch (Exception $exception) {
            return response()->json([
                'error' => true,
                'message' => $exception->getMessage()
            ], 404);
        }
    }

    /**
     * @param string $tabId
     * @return JsonResponse
     */
    public function items($tabId)
    {
        try {
            return response()->json($this->service->getItems($tabId));
        } catch (Exception $exception) {
            return response()->json([
                'error' => true,
                'message' => $exception->getMessage()
            ], 500);
        }
    }

    /**
     * @param Request $request
     * @param string  $tabId
     * @return JsonResponse|Response
     */
    public function destroy(Request $request, string $tabId)
    {
        try {
            $this->service->delete($tabId, $request->all());
            return response()->noContent(200);
        } catch (UnauthorizedException $unauthorizedException) {
            return response()->json([
                'error' => true,
                'message' => $unauthorizedException->getMessage()
            ], 403);
        } catch (ModelNotFoundException $unauthorizedException) {
            return response()->json([
                'error' => true,
                'message' => $unauthorizedException->getMessage()
            ], 404);
        } catch (Exception $exception) {
            return response()->json([
                'error' => true,
                'message' => $exception->getMessage()
            ], 500);
        }
    }

    /**
     * @param Request $request
     * @param string  $tabId
     * @return JsonResponse|Response
     */
    public function block(Request $request, string $tabId)
    {
        try {
            $this->service->lock($tabId, $request->all());
            return response()->noContent(200);
        } catch (UnauthorizedException $unauthorizedException) {
            return response()->json([
                'error' => true,
                'message' => $unauthorizedException->getMessage()
            ], 403);
        } catch (ModelNotFoundException $unauthorizedException) {
            return response()->json([
                'error' => true,
                'message' => $unauthorizedException->getMessage()
            ], 404);
        } catch (Exception $exception) {
            return response()->json([
                'error' => true,
                'message' => $exception->getMessage()
            ], 500);
        }
    }

    /**
     * @param Request $request
     * @param string  $tabId
     * @return JsonResponse|Response
     */
    public function unblock(Request $request, string $tabId)
    {
        try {
            $this->service->unlock($tabId, $request->all());
            return response()->noContent(200);
        } catch (UnauthorizedException $unauthorizedException) {
            return response()->json([
                'error' => true,
                'message' => $unauthorizedException->getMessage()
            ], 403);
        } catch (ModelNotFoundException $unauthorizedException) {
            return response()->json([
                'error' => true,
                'message' => $unauthorizedException->getMessage()
            ], 404);
        } catch (Exception $exception) {
            return response()->json([
                'error' => true,
                'message' => $exception->getMessage()
            ], 500);
        }
    }
}
