<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

use App\Http\Requests;
use Prettus\Validator\Contracts\ValidatorInterface;
use Prettus\Validator\Exceptions\ValidatorException;
use App\Http\Requests\TurnosCreateRequest;
use App\Http\Requests\TurnosUpdateRequest;
use App\Repositories\TurnosRepository;
use App\Validators\TurnosValidator;

/**
 * Class TurnosController.
 *
 * @package namespace App\Http\Controllers;
 */
class TurnosController extends Controller
{
    /**
     * @var TurnosRepository
     */
    protected $repository;

    /**
     * @var TurnosValidator
     */
    protected $validator;

    /**
     * TurnosController constructor.
     *
     * @param TurnosRepository $repository
     * @param TurnosValidator $validator
     */
    public function __construct(TurnosRepository $repository, TurnosValidator $validator)
    {
        $this->repository = $repository;
        $this->validator = $validator;
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function index()
    {
        $this->repository->pushCriteria(app('Prettus\Repository\Criteria\RequestCriteria'));
        $turnos = $this->repository->all();

        return response()->json($turnos);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param TurnosCreateRequest $request
     *
     * @return \Illuminate\Http\JsonResponse
     *
     * @throws \Prettus\Validator\Exceptions\ValidatorException
     */
    public function store(TurnosCreateRequest $request)
    {
        try {
            $this->validator->with($request->all())->passesOrFail(ValidatorInterface::RULE_CREATE);

            $turno = $this->repository->create($request->all());

            $response = [
                'message' => 'Turnos created.',
                'data' => $turno->toArray(),
            ];

            return response()->json($response);

        } catch (ValidatorException $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessageBag()
            ]);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param int $id
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function show($id)
    {
        $turno = $this->repository->find($id);

        return response()->json([
            'data' => $turno,
        ]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param TurnosUpdateRequest $request
     * @param string $id
     *
     * @return \Illuminate\Http\JsonResponse
     *
     * @throws \Prettus\Validator\Exceptions\ValidatorException
     */
    public function update(TurnosUpdateRequest $request, $id)
    {
        try {

            $this->validator->with($request->all())->passesOrFail(ValidatorInterface::RULE_UPDATE);

            $turno = $this->repository->update($request->all(), $id);

            $response = [
                'message' => 'Turnos updated.',
                'data' => $turno->toArray(),
            ];

            return response()->json($response);

        } catch (ValidatorException $e) {

            return response()->json([
                'error' => true,
                'message' => $e->getMessageBag()
            ]);
        }
    }


    /**
     * Remove the specified resource from storage.
     *
     * @param int $id
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id)
    {
        $deleted = $this->repository->delete($id);

        return response()->json([
            'message' => 'Turnos deleted.',
            'deleted' => $deleted,
        ]);
    }
}
