<?php

namespace App\Http\Controllers;

use App\Entities\Logs;
use App\Http\Controllers\Controller;
use App\Services\Nati\AliquotaService;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\JsonResponse;
use App\Http\Requests\Nati\AliquotaCreateRequest;
use App\Http\Requests\Nati\AliquotaUpdateRequest;
use Prettus\Validator\Exceptions\ValidatorException;


class AliquotasController extends Controller
{
    protected $service;
    protected $user;

    public function __construct(AliquotaService $service)
    {
        $this->service = $service;
        $this->user = auth()->user();
    }

    /**
     * Display a listing of the resource.
     *
     * @return JsonResponse
     */
    public function index()
    {
        try {
            $aliquotas = $this->service->all();

            return response()->json($aliquotas);
        } catch (\Exception $exception) {
            return response()->json([
                'error' => true,
                'message' => $exception->getMessage()
            ], 404);
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param AliquotaCreateRequest $request
     *
     * @return JsonResponse
     */
    public function store(AliquotaCreateRequest $request)
    {
        try {

            $aliquota = $this->service->create($request->all());
            $user = auth()->user();
            Logs::writeLog($user->stApelido, "ALIQUOTA", 'O USUARIO [' . $user->stApelido . '] CADASTROU A ALIQUOTA [' . $aliquota->stAliquota . ']');

            $response = [
                'message' => 'Aliquota created.',
                'data' => $aliquota->toArray(),
            ];


            return response()->json($response, 201);

        } catch (ModelNotFoundException $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessageBag()
            ], 400);
        } catch (ValidatorException $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessageBag()
            ], 404);
        }
    }


    /**
     * Display the specified resource.
     *
     * @param int $id
     *
     * @return JsonResponse
     */
    public function show($id)
    {
        try {
            return response()->json($this->service->getById($id));
        } catch (\Exception $exception) {
            return response()->json([
                'error' => true,
                'message' => $exception->getMessage()
            ], 404);
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * @param AliquotaUpdateRequest $request
     * @param string                $id
     *
     * @return JsonResponse
     */
    public function update(AliquotaUpdateRequest $request, $id)
    {
        try {
            $aliquota = $this->service->update($request->all(), $id);
            $user = auth()->user();
            Logs::writeLog($user->stApelido, "ALIQUOTA", 'O USUARIO [' . $user->stApelido . '] ALTEROU A ALIQUOTA [' . $aliquota->stAliquota . ']');

            return response()->json([
                'message' => 'Aliquota Update',
                'data' => $aliquota
            ], 201);


        } catch (ModelNotFoundException $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessage()
            ], 400);

        } catch (ValidatorException $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessageBag()
            ], 404);
        }
    }


    /**
     * Remove the specified resource from storage.
     *
     * @param int $id
     *
     * @return JsonResponse
     */
    public function destroy($id)
    {
        try {
            $aliquota = $this->service->delete($id);

            $user = auth()->user();
            Logs::writeLog($user->stApelido, "ALIQUOTA", 'O USUARIO [' . $user->stApelido . '] DELETOU A ALIQUOTA [' . $aliquota->stAliquota . ']');
            $response = [
                'message' => 'Aliquota deletada.',
                'data' => $aliquota,
            ];
            return response()->json($response);

        } catch (\Exception $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessage()
            ], 400);
        }
    }

    public function transfer($idAliquota, $idAliquotaNew)
    {
        try {
            $aliquota = $this->service->transfer($idAliquota, $idAliquotaNew);

            $response = [
                'message' => 'Aliquotas transferidas com sucesso.',
                'data' => $aliquota,
            ];
            return response()->json($response);

        } catch (\Exception $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessage()
            ], 400);
        }
    }
}
