<?php

namespace App\Http\Controllers;

use App\Entities\Logs;
use App\Http\Requests\CustomerCreateRequest;
use App\Http\Requests\CustomerUpdateRequest;
use App\Services\CustomerService;
use http\Exception;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Prettus\Validator\Exceptions\ValidatorException;
use App\Entities\Nati\Customer;

class CustomersController extends Controller
{
    /**
     * @var CustomerService
     */
    protected $service;


    public function __construct(CustomerService $service)
    {
        $this->service = $service;
    }


    /**
     * Display a listing of the resource.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function index(Request $request)
    {
//        return view('delivery.clientes.index');
        try {
            return response()->json($this->service->get($request->query()));

        } catch (\Exception $exception) {
            return response()->json([
                'error' => true,
                'message' => $exception->getMessage()
            ], 400);
        }
    }

    /**
     * get customer by phone or name
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function filter(Request $request)
    {
        try {
            return response()->json($this->service->filter($request->all()));
        } catch (\Exception $exception) {
            return response()->json([
                'error' => true,
                'message' => $exception->getMessage()
            ], 400);
        }
    }

    public function getNextId()
    {
        return response()->json($this->service->getNextId());
    }

    /**
     * Store a newly created resource in storage.
     * @param CustomerCreateRequest $request
     * @return JsonResponse
     */
    public function store(CustomerCreateRequest $request)
    {
        try {
            $user = auth()->user();
            $response = $this->service->insert($request->all());
            Logs::writeLog($user->stApelido, "CLIENTES", 'O USUARIO [' . $user->stApelido . '] CADASTROU O CLIENTE [' . $response->stCliente . ']');
        }catch (ValidatorException $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessageBag()
            ], 400);
        }
        catch (\Exception $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessage()
            ], 400);
        }
    }

    public function count(){
        try {
            return response()->json($this->service->count());
        } catch (Exception $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessage()
            ], 400);
        }
    }

    /**
     * Show the specified resource.
     * @param $id
     * @return JsonResponse
     */
    public function show($id)
    {
        try {
            return $this->service->getById($id);

        } catch (\Exception $exception) {
            return response()->json([
                'error' => true,
                'message' => $exception->getMessage()
            ], 404);
        }
    }

    public function getOnlyCustomer($id)
    {
        try {
            $customer = $this->service->getOnlyCustomer($id);
            return $customer;
        } catch (\Exception $exception) {
            return response()->json([
                'error' => true,
                'message' => $exception->getMessage()
            ], 404);
        }
    }

    /**
     * get customer by phone or name
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function search(Request $request)
    {
        try {
            return response()->json($this->service->search($request->query()));
        } catch (\Exception $exception) {
            return response()->json([
                'error' => true,
                'message' => $exception->getMessage()
            ], 400);
        }
    }

    /**
     * Update the specified resource in storage.
     * @param CustomerUpdateRequest $request
     * @param                       $id
     * @return JsonResponse
     */
    public function update(CustomerUpdateRequest $request, $id)
    {
        try {
            $user = auth()->user();
            $response = $this->service->update($request->all(), $id);
            Logs::writeLog($user->stApelido, "CLIENTES", 'O USUARIO [' . $user->stApelido . '] EDITOU O CLIENTE [' . $id . ' - ' . $response->stCliente . ']');

        } catch (ValidatorException $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessageBag()
            ], 400);

        } catch (ModelNotFoundException $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessage()
            ], 404);
        }
    }

    /**
     * Remove the specified resource from storage.
     * @param $id
     * @return JsonResponse
     */
    public function destroy($id)
    {
        try {
            $user = auth()->user();
            $customer = Customer::where('idCliente', $id)->first();
            $response = $this->service->delete($id);
            Logs::writeLog($user->stApelido, "CLIENTES", 'O OPERADOR [' . $user->stApelido . '] DESATIVOU O CLIENTE [' . $id . ' - ' . $customer->stCliente . ']');

            return response()->json([
                'data' => $response,
                'message' => 'Success'
            ]);

        } catch (ModelNotFoundException $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessage()
            ], 404);
        }
    }

    public function find($text)
    {
        if (strlen($text) > 0) {
            $text = '%' . $text . '%';
            $customer = Customer::where('stCliente', 'like', $text)->get();

            if (count($customer)) {
                return json_encode($customer);

            } else {
                return null;
            }

        }
    }
}
