<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Http\Requests\Nati\KeyCreateRequest;
use App\Http\Requests\Nati\KeyUpdateRequest;
use App\Services\Nati\KeyService;
use Exception;
use Illuminate\Database\Eloquent\ModelNotFoundException;

use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Prettus\Validator\Exceptions\ValidatorException;


class KeysController extends Controller
{

    protected $service;

    /**
     * KeysController constructor.
     *
     * @param KeyService $service
     */
    public function __construct(KeyService $service)
    {
        $this->service = $service;
    }

    /**
     * Display a listing of the resource.
     *
     * @param KeyCreateRequest $request
     *
     * @return JsonResponse
     */
    public function index(KeyCreateRequest $request)
    {
        try {
            $keys = $this->service->all($request);

            return response()->json($keys);
        } catch (Exception $exception) {
            return response()->json([
                'error' => true,
                'message' => $exception->getMessage()
            ], 404);
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param KeyCreateRequest $request
     *
     * @return JsonResponse
     */
    public function store(KeyCreateRequest $request)
    {
        try {
            $key = $this->service->create($request->all());

            $response = [
                'message' => 'Key created.',
                'data' => $key->toArray(),
            ];
            return response()->json($response, 201);

        } catch (ModelNotFoundException $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessageBag()
            ], 400);
        } catch (ValidatorException $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessageBag()
            ], 404);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param int $id
     *
     * @return JsonResponse
     */
    public function show($id)
    {
        try {
            $key = $this->service->getById($id);

            return response()->json($key);
        } catch (Exception $exception) {
            return response()->json([
                'error' => true,
                'message' => $exception->getMessage()
            ], 404);
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * @param KeyUpdateRequest $request
     * @param string           $id
     *
     * @return JsonResponse
     */
    public function update(KeyUpdateRequest $request, $id)
    {
        try {
            $response = $this->service->update($request->all(), $id);
            return response()->json([
                'message' => 'Key Update',
                'data' => $response
            ], 201);


        } catch (ModelNotFoundException $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessage()
            ], 400);

        } catch (ValidatorException $e) {

            return response()->json([
                'error' => true,
                'message' => $e->getMessageBag()
            ], 404);
        }
    }

    public function unsetKey($id)
    {
        $unset = $this->service->unsetKey($id);
        return $unset;
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param string $id
     *
     * @return JsonResponse
     */
    public function destroy(string $id)
    {
        try {
            $key = $this->service->delete($id);

            $response = [
                'message' => 'Key deletada.',
                'data' => $key,
            ];
            return response()->json($response);

        } catch (Exception $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessage()
            ], 400);
        }
    }

    /**
     * @param Request $request
     *
     * @return JsonResponse
     */
    public function getKeyAvailable(Request $request)
    {
        try {
            $keys = $this->service->keyAvailable($request);

            return response()->json($keys);
        } catch (Exception $exception) {
            return response()->json([
                'error' => true,
                'message' => $exception->getMessage()
            ], 404);
        }
    }

    /**
     * @return JsonResponse
     */
    public function getAvailableComponents()
    {
        try {
            $data = $this->service->getAvailableComponents();
            return response()->json($data);
        } catch (Exception $exception) {
            return response()->json([
                'error' => true,
                'message' => $exception->getMessage()
            ], 400);
        }
    }

    /**
     * @param Request $request
     * @return JsonResponse|Response
     */
    public function import(Request $request){
        try {
            $this->service->import($request->all());
            return response()->noContent();
        } catch (Exception $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessage()
            ], 400);
        }
    }
}
