<?php

namespace App\Services\Nati;

use App\Entities\Nati\Product;
use App\Repositories\Nati\ProductRepository;
use Carbon\Carbon;
use Exception;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Prettus\Validator\Contracts\ValidatorInterface;

class AdjustProductTaxService
{
    protected $repository;
    protected $user;
    protected $aliquotaService;
    /**
     * ProdutoService constructor.
     *
    * @param ProductRepository $repository
    */
    public function __construct(
        ProductRepository $repository,
        AliquotaService $aliquotaService
    )
    {
        $this->repository = $repository;
        $this->user = auth()->user();
        $this->aliquotaService = $aliquotaService;
    }
    public function getProductsWithTaxByCategory($params)
    {
        $order = !$params['order'] ? 'ASC' : Str::upper($params['order']);
        $idCategory = $params['categoria'];

       return Product::selectRaw(
            'prd_codigos.idCodigo,
                prd_produtos.idProduto,
                prd_produtos.stProduto,
                prd_produtos.idCategoria,
                prd_produtos.nrNCM,
                prd_produtos.stCest,
                prd_aliquotas.idAliquota,
                prd_aliquotas.stAliquota,
                prd_aliquotas.vrAliquota,
                prd_aliquotas.codigo,
                prd_aliquotas.cstIcms,
                prd_aliquotas.cstPis,
                prd_aliquotas.cstCofins,
                prd_aliquotas.cfopEntrada,
                prd_aliquotas.cfopSaida,
                prd_aliquotas.iva,
                prd_aliquotas.aliquotaIcms,
                prd_aliquotas.aliquotaCofins,
                prd_aliquotas.aliquotaPis')
        ->join('prd_codigos', 'prd_produtos.idProduto', '=', 'prd_codigos.idProduto')
        ->join('prd_aliquotas', 'prd_produtos.idAliquota', '=', 'prd_aliquotas.idAliquota')
        ->where('prd_produtos.idCategoria', $idCategory)
        ->orderBy('prd_Produtos.dtAlteracao', $order)
        ->paginate($params['limit'], $params['page']);

    }
    public function getProductsWithTaxByFilter($params)
    {
        $order = !$params['order'] ? 'ASC' : Str::upper($params['order']);

       return Product::selectRaw(
            'prd_codigos.idCodigo,
                prd_produtos.idProduto,
                prd_produtos.stProduto,
                prd_produtos.idCategoria,
                prd_produtos.nrNCM,
                prd_produtos.stCest,
                prd_aliquotas.idAliquota,
                prd_aliquotas.stAliquota,
                prd_aliquotas.vrAliquota,
                prd_aliquotas.codigo,
                prd_aliquotas.cstIcms,
                prd_aliquotas.cstPis,
                prd_aliquotas.cstCofins,
                prd_aliquotas.cfopEntrada,
                prd_aliquotas.cfopSaida,
                prd_aliquotas.iva,
                prd_aliquotas.aliquotaIcms,
                prd_aliquotas.aliquotaCofins,
                prd_aliquotas.aliquotaPis')
        ->join('prd_codigos', 'prd_produtos.idProduto', '=', 'prd_codigos.idProduto')
        ->join('prd_aliquotas', 'prd_produtos.idAliquota', '=', 'prd_aliquotas.idAliquota')
        ->where('prd_produtos.stProduto', 'like', "%{$params['filter']}%")
        ->orderBy('prd_Produtos.stProduto', $order)
        ->paginate($params['limit'], $params['page']);
    }

    public function updateProductsTax($products)
    {
        if(count($products) > 0){
            foreach ($products as $product) {
                $tax = $this->getTax($product);
                $product['idAliquota'] = $tax['idAliquota'];
                $product['dtUltimaAlteracao'] = Carbon::now()->toDateTimeString();
                $product['dtAlteracao'] = Carbon::now()->toDateTimeString();
                $product['stOperador'] = auth()->user()->stApelido;
                try {
                    $this->repository->update($product, $product['idProduto']);
                } catch (Exception $e) {
                    throw new Exception("Erro ao atualizar o produto id - {$product['idProduto']}");
                }
            }
        }
    }

    private function getTax($product)
    {
        $tax = [];
        $tax['CST-ICMS'] = str_pad(utf8_encode($product['cstIcms']), 2, '0', STR_PAD_LEFT);
        $tax['CST-PIS'] = str_pad($product['cstPis'], 2, '0', STR_PAD_LEFT);
        $tax['CST-COFINS'] = str_pad($product['cstCofins'], 2, '0', STR_PAD_LEFT);
        $tax['CFOP'] = $product['cfopSaida'];
        $tax['ALIQ-ICMS'] = $product['aliquotaIcms'];
        $tax['ALIQ-COFINS'] = $product['aliquotaCofins'];
        $tax['ALIQ-PIS'] = $product['aliquotaPis'];
        try {
            return $this->aliquotaService->getOrCreate($tax);
        } catch (Exception $e) {
            throw new Exception("Erro ao atualizar a aliquota do produto {$product['idProduto']}");
        }
    }
}
