<?php

namespace App\Services\Nati;

use App\Entities\Delivery\DlvConfig;
use App\Entities\Nati\Key;
use App\Entities\Nati\MicroTerminal;
use App\Repositories\Nati\KeyRepository;
use App\Validators\Nati\KeyValidator;
use Prettus\Validator\Contracts\ValidatorInterface;
use Carbon\Carbon;

class KeyService
{
    private $repository;
    private $validator;


    public function __construct(KeyRepository $repository, KeyValidator $validator)
    {
        $this->repository = $repository;
        $this->validator = $validator;
    }

    public function getById($id)
    {
        return $this->repository->find($id);
    }

    public function findWhere($key)
    {
        return $this->repository->findWhere(['stChave' => $key]);
    }

    public function all($request)
    {
        if (strlen($request->tipo) > 0) {

            return $this->repository->findWhere([
                'stComponente' => null,
                'stTipo' => $request->tipo
            ]);
        }

        return $this->repository->all();
    }

    public function create($request)
    {
        $this->validator->with($request)->passesOrFail(ValidatorInterface::RULE_CREATE);

        $key = [
            "stIndex" => $request['stIndex'],
            "stChave" => $request['stChave'],
            "stData" => Carbon::createFromFormat("d/m/Y", $request['stData'])->format("Ymd"),
            "stTipo" => $request['stTipo'],
            "stComponente" => $request['stComponente'],
            "dtAlteracao" => Carbon::now(),
        ];

        return $this->repository->create($key);
    }

    public function update($request, $id)
    {
        $this->validator->with($request)->passesOrFail(ValidatorInterface::RULE_UPDATE);

        $key = [
            "stIndex" => $request['stIndex'],
            "stChave" => $request['stChave'],
            "stData" => Carbon::createFromFormat("d/m/Y", $request['stData'])->format("Ymd"),
            "stTipo" => $request['stTipo'],
            "stComponente" => $request['stComponente'],
            "dtAlteracao" => Carbon::now(),
        ];

        return $this->repository->update($key, $id);
    }

    public function setKey($key, $id_delivery)
    {
        $model = $this->getById($key);

        return $model->update(['stComponente' => $id_delivery]);
    }

    public function unsetKey($key)
    {
        $model = $this->getById($key);

        return $model->update(['stComponente' => null]);
    }

    public function delete($id)
    {
        return $this->repository->delete($id);
    }

    public function keyAvailable($request)
    {
        return $this->repository->findWhere([
            'type' => $request->type,
            'stComponente' => null
        ]);
    }

    public function getAvailableComponents()
    {
        $delivery = DlvConfig::leftJoin('nat_Keys', 'dlv_Config.idDelivery', '=', 'nat_Keys.stComponente')
            ->whereNull('nat_Keys.stComponente')
            ->get()
            ->map(function ($item) {
                return $item->idDelivery;
            });

        $terminal = MicroTerminal::leftJoin('nat_Keys', 'mt_Microterminais.idMicroTerminal', '=', 'nat_Keys.stComponente')
            ->whereNull('nat_Keys.stComponente')
            ->get()
            ->map(function ($item) {
                return $item->idMicroTerminal;
            });

        $caixa = Key::where('stTipo', 'CX')
            ->whereNotNull('nat_Keys.stComponente')
            ->orderBy('nat_Keys.stComponente')
            ->get()
            ->map(function ($item) {
                return intval($item->stComponente);
            });

        $caixasDisponiveis = array_diff(range(1, 20), $caixa->toArray());

        return [
            'caixas' => array_keys($caixasDisponiveis),
            'terminal' => $terminal,
            'delivery' => $delivery
        ];
    }

    public function import(array $request)
    {
        foreach ($request['keys'] as $key){
            $this->repository->updateOrCreate($key);
        }
    }
}
