<?php

namespace App\Services\Nati;

use App\Entities\RespostaObservacao;
use App\Repositories\RespostaObservacaoRepository;
use App\Services\Service;
use App\Validators\RespostaObservacaoValidator;
use Illuminate\Support\Str;
use Prettus\Validator\Contracts\ValidatorInterface;
use Carbon\Carbon;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;

class RespostaObservacaoService extends Service
{
    private $repository;
    private $validator;

    /**
     * CategoriaService constructor.
     * @param RespostaObservacaoRepository $repository
     * @param RespostaObservacaoValidator $validator
     */
    public function __construct(RespostaObservacaoRepository $repository, RespostaObservacaoValidator $validator)
    {
        $this->repository = $repository;
        $this->validator = $validator;
    }


    public function getAll($request)
    {
        if($request['filter']) {
            return RespostaObservacao::with('respostas')
                ->with('perguntas')
                ->where('stObservacao', 'LIKE', "%$request[filter]%")
                ->where('flAtivo', true)
                ->orderBy('stObservacao','asc')
                ->paginate(10);
        }

        return RespostaObservacao::with('respostas')
        ->with('perguntas')
        ->where('flAtivo', true)
        ->orderBy('stObservacao','asc')
        ->paginate(10);

    }


    public function store($data)
    {
        $this->validator->with($data)->passesOrFail(ValidatorInterface::RULE_CREATE);
        $arr = [
            'stObservacao' => strtoupper($data['stObservacao']),
            'flAtivo' => 1,
            'dtAlteracao' => Carbon::now(),
        ];

        $observacao = $this->repository->create($arr);
        return $observacao;
    }

    public function update($request, $id)
    {
        $this->validator->with($request)->passesOrFail(ValidatorInterface::RULE_UPDATE);

        $request['stObservacao'] = Str::upper($request['stObservacao']);
        $result = $this->repository->update($request, $id);
        return $result;
    }


    public function delete($id)
    {
        $observacao = $this->repository->find($id);

        if ($observacao->respostas->count() > 0) {
            throw new BadRequestHttpException('Não é possivel desativar uma observação que está vinculada a respostas');
        } else {
            $observacao->flAtivo = false;
            $observacao->save();
            return $observacao;
        }
    }

    public function filter($data)
    {
        return RespostaObservacao::where('stObservacao', 'LIKE', "%$data%")
            ->where('flAtivo', true)
            ->paginate(10);
    }
    public function show($id)
    {
        return $this->repository->with('respostas')->with('perguntas')->find($id);
    }

}
