<?php

namespace App\Services;


use Carbon\Carbon;
use App\Entities\Supplier;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Str;
use App\Validators\FornecedorValidator;
use App\Repositories\FornecedorRepository;
use Prettus\Validator\Contracts\ValidatorInterface;

class SuppliersService {

    protected $repository;
    protected $validator;

    public function __construct(FornecedorRepository $repository, FornecedorValidator $validator)
    {
        $this->repository = $repository;
        $this->validator = $validator;
    }

    public function get($request)
    {
        $key = md5(json_encode($request));
        if (Cache::tags(['ListaSupplier'])->has($key)) {
            return Cache::tags(['ListaSupplier'])->get($key);
        } else {
            $order = !$request['order'] ? 'ASC' : Str::upper($request['order']);
            $response = Supplier::select(['*'])
                ->where('flAtivo', true)
                ->where('idFornecedor', '<>', '0')
                ->where(function ($query) use ($request) {
                    if ($request['filter'] != null and strlen($request['filter']) > 0) {
                        $query->where('stNomeFornecedor', 'LIKE', '%' . $request['filter'] . '%')
                            ->orWhere('stCNPJ', 'LIKE', '%' . $request['filter'] . '%');
                    }
                })
                ->orderBy($request['sort'], $order)
                ->paginate($request['limit'], $request['page']);

            Cache::tags(['ListaSupplier'])->put($key, $response, 1440);
            return $response;
        }
    }

    public function getById($id)
    {
        return $this->repository->find($id);
    }

    public function search($request)
    {
        return Supplier::where('stCNPJ', 'LIKE', '%'.$request->search.'%')
            ->orWhere('stNomeFornecedor', 'LIKE', '%'.$request->search.'%')
            ->get();
    }

    public function insert($data)
    {
        $data['idFornecedor'] = Supplier::max('idFornecedor') + 1;
        $data['flAtivo'] = true;
        $data['dtAlteracao'] = Carbon::now();
        $data['stNomeFornecedor'] = Str::upper($data['stNomeFornecedor']);

        $this->validator->with($data)->passesOrFail(ValidatorInterface::RULE_CREATE);
        $this->repository->create($data);
        return $data;
    }

    public function update($data, $id)
    {
        $data['active'] = true;
        $data['last_change'] = Carbon::now();

        $this->validator->with($data)->passesOrFail(ValidatorInterface::RULE_UPDATE);
        return $this->repository->update($data, $id);
    }

    public function delete($id)
    {
        return $this->repository->update(['flAtivo' => false], $id);
//        return Supplier::find($id)->update(['active' => false]);
    }

    public function count()
    {
        if (Cache::has('suppliers_count')) {
            return Cache::get('suppliers_count');
        }else{
            $count =$this->repository->findByField('flAtivo', true)->count();
            Cache::put('suppliers_count', $count);
            return $count;
        }
    }

}
